package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request responsible user DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request responsible user.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestResponsibleUserDto {

    private final String responsible;
    private final List<Long> serviceIds;

    @JsonCreator
    public TransferRequestResponsibleUserDto(String responsible, List<Long> serviceIds) {
        this.responsible = responsible;
        this.serviceIds = serviceIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Responsible user uid.", required = true)
    public String getResponsible() {
        return responsible;
    }

    @Schema(description = "Matching service ids.", required = true)
    public List<Long> getServiceIds() {
        return serviceIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestResponsibleUserDto that = (TransferRequestResponsibleUserDto) o;
        return Objects.equals(responsible, that.responsible) &&
                Objects.equals(serviceIds, that.serviceIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsible, serviceIds);
    }

    @Override
    public String toString() {
        return "TransferRequestResponsibleUserDto{" +
                "responsible='" + responsible + '\'' +
                ", serviceIds=" + serviceIds +
                '}';
    }

    public static final class Builder {

        private final List<Long> serviceIds = new ArrayList<>();

        private String responsible;

        private Builder() {
        }

        public Builder responsible(String responsible) {
            this.responsible = responsible;
            return this;
        }

        public Builder addServiceId(long serviceId) {
            this.serviceIds.add(serviceId);
            return this;
        }

        public Builder addServiceIds(Long... serviceIds) {
            this.serviceIds.addAll(Arrays.asList(serviceIds));
            return this;
        }

        public Builder addServiceIds(Collection<? extends Long> serviceIds) {
            this.serviceIds.addAll(serviceIds);
            return this;
        }

        public TransferRequestResponsibleUserDto build() {
            Objects.requireNonNull(responsible, "Responsible is required");
            return new TransferRequestResponsibleUserDto(responsible, serviceIds);
        }

    }

}
