package ru.yandex.intranet.d.web.model.transfers.api;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestVoteTypeDto;

/**
 * Transfer request voter DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request voter.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestVoterDto {

    private final String voter;
    private final List<String> folderIds;
    private final List<Long> serviceIds;
    private final Instant timestamp;
    private final TransferRequestVoteTypeDto voteType;
    private final List<String> providerIds;

    @JsonCreator
    public TransferRequestVoterDto(
            String voter,
            List<String> folderIds,
            List<Long> serviceIds,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant timestamp,
            TransferRequestVoteTypeDto voteType,
            List<String> providerIds) {
        this.voter = voter;
        this.folderIds = folderIds;
        this.serviceIds = serviceIds;
        this.timestamp = timestamp;
        this.voteType = voteType;
        this.providerIds = providerIds != null ? providerIds : List.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Voter uid.", required = true)
    public String getVoter() {
        return voter;
    }

    @Schema(description = "Matching folder ids.", required = true)
    public List<String> getFolderIds() {
        return folderIds;
    }

    @Schema(description = "Matching service ids.", required = true)
    public List<Long> getServiceIds() {
        return serviceIds;
    }

    @Schema(description = "Vote timestamp.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getTimestamp() {
        return timestamp;
    }

    @Schema(description = "Vote type.", required = true)
    public TransferRequestVoteTypeDto getVoteType() {
        return voteType;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Provider ids when user is responsible for those providers.")
    public List<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestVoterDto that = (TransferRequestVoterDto) o;
        return Objects.equals(voter, that.voter) &&
                Objects.equals(folderIds, that.folderIds) &&
                Objects.equals(serviceIds, that.serviceIds) &&
                Objects.equals(timestamp, that.timestamp) &&
                voteType == that.voteType &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(voter, folderIds, serviceIds, timestamp, voteType, providerIds);
    }

    @Override
    public String toString() {
        return "TransferRequestVoterDto{" +
                "voter='" + voter + '\'' +
                ", folderIds=" + folderIds +
                ", serviceIds=" + serviceIds +
                ", timestamp=" + timestamp +
                ", voteType=" + voteType +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final List<String> folderIds = new ArrayList<>();
        private final List<Long> serviceIds = new ArrayList<>();
        private final List<String> providerIds = new ArrayList<>();

        private String voter;
        private Instant timestamp;
        private TransferRequestVoteTypeDto voteType;

        private Builder() {
        }

        public Builder voter(String voter) {
            this.voter = voter;
            return this;
        }

        public Builder addFolderId(String folderId) {
            this.folderIds.add(folderId);
            return this;
        }

        public Builder addFolderIds(String... folderIds) {
            this.folderIds.addAll(Arrays.asList(folderIds));
            return this;
        }

        public Builder addFolderIds(Collection<? extends String> folderIds) {
            this.folderIds.addAll(folderIds);
            return this;
        }

        public Builder addServiceId(long serviceId) {
            this.serviceIds.add(serviceId);
            return this;
        }

        public Builder addServiceIds(Long... serviceIds) {
            this.serviceIds.addAll(Arrays.asList(serviceIds));
            return this;
        }

        public Builder addServiceIds(Collection<? extends Long> serviceIds) {
            this.serviceIds.addAll(serviceIds);
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Builder voteType(TransferRequestVoteTypeDto voteType) {
            this.voteType = voteType;
            return this;
        }

        public Builder addProviderId(String providerId) {
            this.providerIds.add(providerId);
            return this;
        }

        public Builder addProviderIds(String... providerIds) {
            this.providerIds.addAll(Arrays.asList(providerIds));
            return this;
        }

        public Builder addProviderIds(Collection<? extends String> providerIds) {
            this.providerIds.addAll(providerIds);
            return this;
        }

        public TransferRequestVoterDto build() {
            Objects.requireNonNull(voter, "Voter is required");
            Objects.requireNonNull(timestamp, "Timestamp is required");
            Objects.requireNonNull(voteType, "VoteType is required");
            return new TransferRequestVoterDto(voter, folderIds, serviceIds, timestamp, voteType, providerIds);
        }

    }

}
