package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Quota transfer parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Quota transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontCreateQuotaTransferDto {

    private final String destinationFolderId;
    private final String destinationServiceId;
    private final List<FrontCreateQuotaResourceTransferDto> resourceTransfers;

    @JsonCreator
    public FrontCreateQuotaTransferDto(
            @JsonProperty("destinationFolderId") String destinationFolderId,
            @JsonProperty("destinationServiceId") String destinationServiceId,
            @JsonProperty("resourceTransfers") List<FrontCreateQuotaResourceTransferDto> resourceTransfers) {
        this.destinationFolderId = destinationFolderId;
        this.destinationServiceId = destinationServiceId;
        this.resourceTransfers = resourceTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Destination folder id.", required = true)
    public Optional<String> getDestinationFolderId() {
        return Optional.ofNullable(destinationFolderId);
    }

    @Schema(description = "Destination service id.", required = true)
    public Optional<String> getDestinationServiceId() {
        return Optional.ofNullable(destinationServiceId);
    }

    @Schema(description = "Quota resource transfers.", required = true)
    public Optional<List<FrontCreateQuotaResourceTransferDto>> getResourceTransfers() {
        return Optional.ofNullable(resourceTransfers);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontCreateQuotaTransferDto that = (FrontCreateQuotaTransferDto) o;
        return Objects.equals(destinationFolderId, that.destinationFolderId) &&
                Objects.equals(destinationServiceId, that.destinationServiceId) &&
                Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(destinationFolderId, destinationServiceId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "FrontCreateQuotaTransferDto{" +
                "destinationFolderId='" + destinationFolderId + '\'' +
                ", destinationServiceId='" + destinationServiceId + '\'' +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static final class Builder {

        private final List<FrontCreateQuotaResourceTransferDto> resourceTransfers = new ArrayList<>();

        private String destinationFolderId;
        private String destinationServiceId;

        private Builder() {
        }

        public Builder destinationFolderId(String destinationFolderId) {
            this.destinationFolderId = destinationFolderId;
            return this;
        }

        public Builder destinationServiceId(String destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public Builder addResourceTransfer(FrontCreateQuotaResourceTransferDto resourceTransfer) {
            this.resourceTransfers.add(resourceTransfer);
            return this;
        }

        public FrontCreateQuotaTransferDto build() {
            return new FrontCreateQuotaTransferDto(destinationFolderId, destinationServiceId, resourceTransfers);
        }

    }

}
