package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Reverse transfer parameters DTO.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Schema(description = "Reserve transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontCreateReserveTransferDto {
    private final String providerId;
    private final String destinationFolderId;
    private final String destinationServiceId;
    private final List<FrontCreateQuotaResourceTransferDto> resourceTransfers;

    @JsonCreator
    public FrontCreateReserveTransferDto(String providerId, String destinationFolderId, String destinationServiceId,
                                         List<FrontCreateQuotaResourceTransferDto> resourceTransfers) {
        this.providerId = providerId;
        this.destinationFolderId = destinationFolderId;
        this.destinationServiceId = destinationServiceId;
        this.resourceTransfers = resourceTransfers;
    }

    @Schema(description = "Provider id.", required = true)
    public Optional<String> getProviderId() {
        return Optional.ofNullable(providerId);
    }

    @Schema(description = "Destination folder id.", required = true)
    public Optional<String> getDestinationFolderId() {
        return Optional.ofNullable(destinationFolderId);
    }

    @Schema(description = "Destination service id.", required = true)
    public Optional<String> getDestinationServiceId() {
        return Optional.ofNullable(destinationServiceId);
    }

    @Schema(description = "Quota resource transfers.", required = true)
    public Optional<List<FrontCreateQuotaResourceTransferDto>> getResourceTransfers() {
        return Optional.ofNullable(resourceTransfers);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontCreateReserveTransferDto that = (FrontCreateReserveTransferDto) o;
        return Objects.equals(providerId, that.providerId) && Objects.equals(destinationFolderId,
                that.destinationFolderId) && Objects.equals(destinationServiceId, that.destinationServiceId)
                && Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, destinationFolderId, destinationServiceId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "FrontCreateReserveTransferDto{" +
                "providerId='" + providerId + '\'' +
                ", destinationFolderId='" + destinationFolderId + '\'' +
                ", destinationServiceId='" + destinationServiceId + '\'' +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private final List<FrontCreateQuotaResourceTransferDto> resourceTransfers = new ArrayList<>();

        private String destinationFolderId;
        private String destinationServiceId;
        private String providerId;

        private Builder() {

        }

        public Builder destinationFolderId(String destinationFolderId) {
            this.destinationFolderId = destinationFolderId;
            return this;
        }

        public Builder destinationServiceId(String destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public Builder addResourceTransfer(FrontCreateQuotaResourceTransferDto resourceTransfer) {
            this.resourceTransfers.add(resourceTransfer);
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public FrontCreateReserveTransferDto build() {
            return new FrontCreateReserveTransferDto(providerId, destinationFolderId, destinationServiceId,
                    resourceTransfers);
        }
    }
}
