package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;

/**
 * Create transfer request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontCreateTransferRequestDto extends FrontBaseCreateTransferRequestDto {

    private final String description;
    private final Boolean addConfirmation;

    @JsonCreator
    public FrontCreateTransferRequestDto(String description,
                                         TransferRequestTypeDto requestType,
                                         FrontCreateTransferRequestParametersDto parameters,
                                         Boolean addConfirmation,
                                         @Nullable FrontTransferLoanParametersDto loanParameters) {
        super(requestType, parameters, loanParameters);
        this.description = description;
        this.addConfirmation = addConfirmation;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @Schema(description = "'Transfer request is auto-confirmed by author with enough permissions' flag.",
            required = true)
    public Optional<Boolean> getAddConfirmation() {
        return Optional.ofNullable(addConfirmation);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        FrontCreateTransferRequestDto that = (FrontCreateTransferRequestDto) o;
        return Objects.equals(description, that.description) &&
                Objects.equals(addConfirmation, that.addConfirmation);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), description, addConfirmation);
    }

    @Override
    public String toString() {
        return "FrontCreateTransferRequestDto{" +
                "description='" + description + '\'' +
                ", addConfirmation=" + addConfirmation +
                '}';
    }

    public static final class Builder {

        private String description;
        private TransferRequestTypeDto requestType;
        private FrontCreateTransferRequestParametersDto parameters;
        private Boolean addConfirmation;
        @Nullable
        private FrontTransferLoanParametersDto loanParameters;

        private Builder() {
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder requestType(TransferRequestTypeDto requestType) {
            this.requestType = requestType;
            return this;
        }

        public Builder parameters(FrontCreateTransferRequestParametersDto parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder addConfirmation(boolean addConfirmation) {
            this.addConfirmation = addConfirmation;
            return this;
        }

        public Builder loanParameters(@Nullable FrontTransferLoanParametersDto loanParameters) {
            this.loanParameters = loanParameters;
            return this;
        }

        public FrontCreateTransferRequestDto build() {
            return new FrontCreateTransferRequestDto(description, requestType, parameters,
                    addConfirmation, loanParameters);
        }

    }

}
