package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Create transfer request parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request parameters.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontCreateTransferRequestParametersDto {

    private final List<FrontCreateQuotaTransferDto> quotaTransfers;
    private final FrontCreateReserveTransferDto reserveTransfer;
    private final List<FrontCreateProvisionTransferDto> provisionTransfers;

    @JsonCreator
    public FrontCreateTransferRequestParametersDto(
            List<FrontCreateQuotaTransferDto> quotaTransfers,
            FrontCreateReserveTransferDto reserveTransfer,
            List<FrontCreateProvisionTransferDto> provisionTransfers) {
        this.quotaTransfers = quotaTransfers;
        this.reserveTransfer = reserveTransfer;
        this.provisionTransfers = provisionTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Quota transfer requests.")
    public Optional<List<FrontCreateQuotaTransferDto>> getQuotaTransfers() {
        return Optional.ofNullable(quotaTransfers);
    }

    @Schema(description = "Reserve transfer requests.")
    public Optional<FrontCreateReserveTransferDto> getReserveTransfer() {
        return Optional.ofNullable(reserveTransfer);
    }

    @Schema(description = "Provision transfer requests.")
    public Optional<List<FrontCreateProvisionTransferDto>> getProvisionTransfers() {
        return Optional.ofNullable(provisionTransfers);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontCreateTransferRequestParametersDto that = (FrontCreateTransferRequestParametersDto) o;
        return Objects.equals(getQuotaTransfers(), that.getQuotaTransfers()) && Objects.equals(getReserveTransfer(),
                that.getReserveTransfer()) && Objects.equals(getProvisionTransfers(), that.getProvisionTransfers());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getQuotaTransfers(), getReserveTransfer(), getProvisionTransfers());
    }

    @Override
    public String toString() {
        return "FrontCreateTransferRequestParametersDto{" +
                "quotaTransfers=" + quotaTransfers +
                ", reserveTransfer=" + reserveTransfer +
                ", provisionTransfers=" + provisionTransfers +
                '}';
    }

    public static final class Builder {

        private final List<FrontCreateQuotaTransferDto> quotaTransfers = new ArrayList<>();
        private final List<FrontCreateProvisionTransferDto> provisionTransfers = new ArrayList<>();
        private FrontCreateReserveTransferDto reserveTransfer;

        private Builder() {
        }

        public Builder addQuotaTransfer(FrontCreateQuotaTransferDto quotaTransfer) {
            this.quotaTransfers.add(quotaTransfer);
            return this;
        }

        public Builder reserveTransfer(FrontCreateReserveTransferDto reserveTransfer) {
            this.reserveTransfer = reserveTransfer;
            return this;
        }

        public Builder addProvisionTransfer(FrontCreateProvisionTransferDto provisionTransfer) {
            this.provisionTransfers.add(provisionTransfer);
            return this;
        }

        public Builder addProvisionTransfers(Collection<FrontCreateProvisionTransferDto> provisionTransfers) {
            this.provisionTransfers.addAll(provisionTransfers);
            return this;
        }

        public FrontCreateTransferRequestParametersDto build() {
            return new FrontCreateTransferRequestParametersDto(quotaTransfers, reserveTransfer, provisionTransfers);
        }

    }

}
