package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Put transfer request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontPutTransferRequestDto {

    private final String description;
    private final FrontCreateTransferRequestParametersDto parameters;
    private final Boolean addConfirmation;
    @Nullable
    private final FrontTransferLoanParametersDto loanParameters;

    @JsonCreator
    public FrontPutTransferRequestDto(String description,
                                      FrontCreateTransferRequestParametersDto parameters,
                                      Boolean addConfirmation,
                                      @Nullable FrontTransferLoanParametersDto loanParameters) {
        this.description = description;
        this.parameters = parameters;
        this.addConfirmation = addConfirmation;
        this.loanParameters = loanParameters;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @Schema(description = "Transfer request parameters.", required = true)
    public Optional<FrontCreateTransferRequestParametersDto> getParameters() {
        return Optional.ofNullable(parameters);
    }

    @Schema(description = "'Transfer request is auto-confirmed by author with enough permissions' flag.",
            required = true)
    public Optional<Boolean> getAddConfirmation() {
        return Optional.ofNullable(addConfirmation);
    }

    @Schema(description = "Loan parameters.")
    public Optional<FrontTransferLoanParametersDto> getLoanParameters() {
        return Optional.ofNullable(loanParameters);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontPutTransferRequestDto that = (FrontPutTransferRequestDto) o;
        return Objects.equals(description, that.description) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(addConfirmation, that.addConfirmation) &&
                Objects.equals(loanParameters, that.loanParameters);
    }

    @Override
    public int hashCode() {
        return Objects.hash(description, parameters, addConfirmation, loanParameters);
    }

    @Override
    public String toString() {
        return "FrontPutTransferRequestDto{" +
                "description='" + description + '\'' +
                ", parameters=" + parameters +
                ", addConfirmation=" + addConfirmation +
                ", loanParameters=" + loanParameters +
                '}';
    }

    public static final class Builder {

        private String description;
        private FrontCreateTransferRequestParametersDto parameters;
        private Boolean addConfirmation;
        @Nullable
        private FrontTransferLoanParametersDto loanParameters;

        private Builder() {
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder parameters(FrontCreateTransferRequestParametersDto parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder addConfirmation(boolean addConfirmation) {
            this.addConfirmation = addConfirmation;
            return this;
        }

        public Builder loanParameters(@Nullable FrontTransferLoanParametersDto loanParameters) {
            this.loanParameters = loanParameters;
            return this;
        }

        public FrontPutTransferRequestDto build() {
            return new FrontPutTransferRequestDto(description, parameters, addConfirmation, loanParameters);
        }

    }

}
