package ru.yandex.intranet.d.web.model.transfers.front

import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import io.swagger.v3.oas.annotations.media.Schema
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.quotas.QuotasHelper
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.model.AmountDto
import java.util.*

/**
 * Quota resource transfer DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Quota resource transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
data class FrontQuotaResourceTransferDto @JsonCreator constructor(
    @get:Schema(description = "Resource id.", required = true)
    val resourceId: String,
    @get:Schema(description = "Transferred amount.", required = true)
    val delta: String,
    @get:Schema(description = "Transferred resource amount unit id.", required = true)
    val deltaUnitId: String,
    @get:Schema(description = "Transferred resource amount unit name.", required = true)
    val deltaUnit: String,
    @get:Schema(description = "Transferred resource delta amount", required = true)
    val deltaAmount: AmountDto
) {

    companion object {
        @JvmStatic
        fun fromModel(
            model: ResourceQuotaTransfer,
            resources: Map<String, ResourceModel>,
            unitsEnsembles: Map<String, UnitsEnsembleModel>,
            locale: Locale
        ): FrontQuotaResourceTransferDto {
            val resource = resources[model.resourceId]
            val unitsEnsemble = unitsEnsembles[resource!!.unitsEnsembleId]
            val deltaAmount = QuotasHelper.getAmountDto(
                QuotasHelper.toBigDecimal(model.delta),
                resource, unitsEnsemble, locale
            )
            val convertedDelta = Units
                .convertToQuotaTransferRequestDisplay(model.delta, resource, unitsEnsemble)
            return FrontQuotaResourceTransferDto(
                resourceId = model.resourceId,
                delta = Units.formatQuotaTransferRequestAmount(convertedDelta.t1, locale),
                deltaUnitId = convertedDelta.t2.id,
                deltaUnit = Units.toShortUnitName(convertedDelta.t1, convertedDelta.t2, locale),
                deltaAmount = deltaAmount,
            )
        }

        @JvmStatic
        fun fromModels(
            transfers: Collection<ResourceQuotaTransfer>,
            resources: Map<String, ResourceModel>,
            unitsEnsembles: Map<String, UnitsEnsembleModel>,
            locale: Locale
        ): List<FrontQuotaResourceTransferDto> = transfers.map {
            fromModel(
                model = it,
                resources = resources,
                unitsEnsembles = unitsEnsembles,
                locale = locale
            )
        }
    }
}
