package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

@Schema(description = "Transfer reserve request responsible.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontReserveResponsibleDto {
    private final Set<String> responsibleIds;
    private final String providerId;
    private final String folderId;
    private final String serviceId;

    @JsonCreator
    public FrontReserveResponsibleDto(Set<String> responsibleIds,
                                      String providerId,
                                      String folderId,
                                      String serviceId) {
        this.responsibleIds = responsibleIds;
        this.providerId = providerId;
        this.folderId = folderId;
        this.serviceId = serviceId;
    }

    @Schema(description = "Provider responsible ids.", required = true)
    public Set<String> getResponsibleIds() {
        return responsibleIds;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Reserve folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Provider service id.", required = true)
    public String getServiceId() {
        return serviceId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontReserveResponsibleDto that = (FrontReserveResponsibleDto) o;
        return Objects.equals(responsibleIds, that.responsibleIds) && Objects.equals(providerId, that.providerId)
                && Objects.equals(folderId, that.folderId) && Objects.equals(serviceId, that.serviceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsibleIds, providerId, folderId, serviceId);
    }

    @Override
    public String toString() {
        return "FrontReserveResponsibleDto{" +
                "responsibleIds=" + responsibleIds +
                ", providerId='" + providerId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", serviceId='" + serviceId + '\'' +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private final Set<String> responsibleIds = new HashSet<>();
        private String providerId;
        private String folderId;
        private String serviceId;

        private Builder() {

        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder serviceId(String serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = String.valueOf(serviceId);
            return this;
        }

        public Builder addResponsibleId(String responsibleId) {
            this.responsibleIds.add(responsibleId);
            return this;
        }

        public Builder addResponsibleIds(String... responsibleId) {
            this.responsibleIds.addAll(Arrays.asList(responsibleId));
            return this;
        }

        public Builder addResponsibleIds(Collection<? extends String> responsibleId) {
            this.responsibleIds.addAll(responsibleId);
            return this;
        }

        public FrontReserveResponsibleDto build() {
            return new FrontReserveResponsibleDto(responsibleIds, Preconditions.checkNotNull(providerId),
                    Preconditions.checkNotNull(folderId), Preconditions.checkNotNull(serviceId));
        }
    }
}
