package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request application details DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request application details.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestApplicationDetailsDto {

    private final FrontTransferRequestApplicationErrorsDto transferErrors;
    private final Map<String, FrontTransferRequestApplicationErrorsDto> transferErrorsByOperationId;
    private final Map<String, FrontTransferOperationStatusDto> operationStatusById;

    @JsonCreator
    public FrontTransferRequestApplicationDetailsDto(
            FrontTransferRequestApplicationErrorsDto transferErrors,
            Map<String, FrontTransferRequestApplicationErrorsDto> transferErrorsByOperationId,
            Map<String, FrontTransferOperationStatusDto> operationStatusById
    ) {
        this.transferErrors = transferErrors;
        this.transferErrorsByOperationId = Objects.requireNonNullElse(transferErrorsByOperationId, Map.of());
        this.operationStatusById = Objects.requireNonNullElse(operationStatusById, Map.of());
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Request application errors.")
    public Optional<FrontTransferRequestApplicationErrorsDto> getTransferErrors() {
        return Optional.ofNullable(transferErrors);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Request application errors by operation id.")
    public Map<String, FrontTransferRequestApplicationErrorsDto> getTransferErrorsByOperationId() {
        return transferErrorsByOperationId;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Operation status by operation id.")
    public Map<String, FrontTransferOperationStatusDto> getOperationStatusById() {
        return operationStatusById;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestApplicationDetailsDto that = (FrontTransferRequestApplicationDetailsDto) o;
        return Objects.equals(transferErrors, that.transferErrors) &&
                Objects.equals(transferErrorsByOperationId, that.transferErrorsByOperationId) &&
                Objects.equals(operationStatusById, that.operationStatusById);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transferErrors, transferErrorsByOperationId, operationStatusById);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestApplicationDetailsDto{" +
                "transferErrors=" + transferErrors +
                ", transferErrorsByOperationId=" + transferErrorsByOperationId +
                ", operationStatusById=" + operationStatusById +
                '}';
    }

    public static final class Builder {
        private final Map<String, FrontTransferRequestApplicationErrorsDto> transferErrorsByOperationId =
                new HashMap<>();
        private final Map<String, FrontTransferOperationStatusDto> operationStatusById = new HashMap<>();

        private FrontTransferRequestApplicationErrorsDto transferErrors;

        private Builder() {
        }

        public Builder transferErrors(FrontTransferRequestApplicationErrorsDto transferErrors) {
            this.transferErrors = transferErrors;
            return this;
        }

        public Builder transferErrorsByOperationId(
                Map<String, FrontTransferRequestApplicationErrorsDto> transferErrorsByOperationId
        ) {
            this.transferErrorsByOperationId.putAll(transferErrorsByOperationId);
            return this;
        }

        public Builder operationStatusById(Map<String, FrontTransferOperationStatusDto> operationStatusById) {
            this.operationStatusById.putAll(operationStatusById);
            return this;
        }

        public FrontTransferRequestApplicationDetailsDto build() {
            return new FrontTransferRequestApplicationDetailsDto(transferErrors, transferErrorsByOperationId,
                    operationStatusById);
        }

    }

}
