package ru.yandex.intranet.d.web.model.transfers.front;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestSubtypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;

/**
 * Transfer request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestDto {

    private final String id;
    private final String version;
    private final String summary;
    private final String description;
    private final String trackerIssueKey;
    private final TransferRequestTypeDto requestType;
    private final TransferRequestStatusDto status;
    private final String createdBy;
    private final String updatedBy;
    private final Instant createdAt;
    private final Instant updatedAt;
    private final Instant appliedAt;
    private final FrontTransferRequestParametersDto parameters;
    private final FrontTransferRequestResponsibleDto transferResponsible;
    private final FrontTransferRequestVotesDto transferVotes;
    private final FrontTransferRequestApplicationDetailsDto applicationDetails;
    private final boolean canUpdate;
    private final boolean canCancel;
    private final boolean canVote;
    private final boolean canProvideOverCommitReserve;
    @Nullable
    private final FrontTransferLoanMetaDto loanMeta;
    @Nullable
    private final TransferRequestSubtypeDto requestSubtype;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public FrontTransferRequestDto(
            @JsonProperty("id") String id,
            @JsonProperty("version") String version,
            @JsonProperty("summary") String summary,
            @JsonProperty("description") String description,
            @JsonProperty("trackerIssueKey") String trackerIssueKey,
            @JsonProperty("requestType") TransferRequestTypeDto requestType,
            @JsonProperty("status") TransferRequestStatusDto status,
            @JsonProperty("createdBy") String createdBy,
            @JsonProperty("updatedBy") String updatedBy,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            @JsonProperty("createdAt") Instant createdAt,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            @JsonProperty("updatedAt") Instant updatedAt,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            @JsonProperty("appliedAt") Instant appliedAt,
            @JsonProperty("parameters") FrontTransferRequestParametersDto parameters,
            @JsonProperty("transferResponsible") FrontTransferRequestResponsibleDto transferResponsible,
            @JsonProperty("transferVotes") FrontTransferRequestVotesDto transferVotes,
            @JsonProperty("applicationDetails") FrontTransferRequestApplicationDetailsDto applicationDetails,
            @JsonProperty("canUpdate") boolean canUpdate,
            @JsonProperty("canCancel") boolean canCancel,
            @JsonProperty("canVote") boolean canVote,
            @JsonProperty("canProvideOverCommitReserve") boolean canProvideOverCommitReserve,
            @JsonProperty("loanMeta") @Nullable FrontTransferLoanMetaDto loanMeta,
            @JsonProperty("requestSubtype") @Nullable TransferRequestSubtypeDto requestSubtype) {
        this.id = id;
        this.version = version;
        this.summary = summary;
        this.description = description;
        this.trackerIssueKey = trackerIssueKey;
        this.requestType = requestType;
        this.requestSubtype = requestSubtype;
        this.status = status;
        this.createdBy = createdBy;
        this.updatedBy = updatedBy;
        this.createdAt = createdAt;
        this.updatedAt = updatedAt;
        this.appliedAt = appliedAt;
        this.parameters = parameters;
        this.transferResponsible = transferResponsible;
        this.transferVotes = transferVotes;
        this.applicationDetails = applicationDetails;
        this.canUpdate = canUpdate;
        this.canCancel = canCancel;
        this.canVote = canVote;
        this.canProvideOverCommitReserve = canProvideOverCommitReserve;
        this.loanMeta = loanMeta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Transfer request version.", required = true)
    public String getVersion() {
        return version;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request summary.")
    public Optional<String> getSummary() {
        return Optional.ofNullable(summary);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request Tracker issue key.")
    public Optional<String> getTrackerIssueKey() {
        return Optional.ofNullable(trackerIssueKey);
    }

    @Schema(description = "Transfer request type.", required = true)
    public TransferRequestTypeDto getRequestType() {
        return requestType;
    }

    @Schema(description = "Transfer request subtype.", required = true)
    public TransferRequestSubtypeDto getRequestSubtype() {
        return requestSubtype;
    }

    @Schema(description = "Transfer request status.", required = true)
    public TransferRequestStatusDto getStatus() {
        return status;
    }

    @Schema(description = "Transfer request author id.", required = true)
    public String getCreatedBy() {
        return createdBy;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request last editor id.")
    public Optional<String> getUpdatedBy() {
        return Optional.ofNullable(updatedBy);
    }

    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request creation timestamp.", required = true)
    public Instant getCreatedAt() {
        return createdAt;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request last update timestamp.")
    public Optional<Instant> getUpdatedAt() {
        return Optional.ofNullable(updatedAt);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request application timestamp.")
    public Optional<Instant> getAppliedAt() {
        return Optional.ofNullable(appliedAt);
    }

    @Schema(description = "Transfer request parameters.", required = true)
    public FrontTransferRequestParametersDto getParameters() {
        return parameters;
    }

    @Schema(description = "Transfer request responsible.", required = true)
    public FrontTransferRequestResponsibleDto getTransferResponsible() {
        return transferResponsible;
    }

    @Schema(description = "Transfer request votes.", required = true)
    public FrontTransferRequestVotesDto getTransferVotes() {
        return transferVotes;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request application details.")
    public Optional<FrontTransferRequestApplicationDetailsDto> getApplicationDetails() {
        return Optional.ofNullable(applicationDetails);
    }

    @Schema(description = "Current user can update this transfer request.", required = true)
    public boolean isCanUpdate() {
        return canUpdate;
    }

    @Schema(description = "Current user can cancel this transfer request.", required = true)
    public boolean isCanCancel() {
        return canCancel;
    }

    @Schema(description = "Current user can vote for this transfer request.", required = true)
    public boolean isCanVote() {
        return canVote;
    }

    @Schema(description = "Current user can provide over commit reserve for this transfer request.", required = true)
    public boolean isCanProvideOverCommitReserve() {
        return canProvideOverCommitReserve;
    }

    @Schema(description = "Loan metadata.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<FrontTransferLoanMetaDto> getLoanMeta() {
        return Optional.ofNullable(loanMeta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestDto that = (FrontTransferRequestDto) o;
        return canUpdate == that.canUpdate &&
                canCancel == that.canCancel &&
                canVote == that.canVote &&
                canProvideOverCommitReserve == that.canProvideOverCommitReserve &&
                Objects.equals(id, that.id) &&
                Objects.equals(version, that.version) &&
                Objects.equals(summary, that.summary) &&
                Objects.equals(description, that.description) &&
                Objects.equals(trackerIssueKey, that.trackerIssueKey) &&
                requestType == that.requestType &&
                requestSubtype == that.requestSubtype &&
                status == that.status &&
                Objects.equals(createdBy, that.createdBy) &&
                Objects.equals(updatedBy, that.updatedBy) &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(updatedAt, that.updatedAt) &&
                Objects.equals(appliedAt, that.appliedAt) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(transferResponsible, that.transferResponsible) &&
                Objects.equals(transferVotes, that.transferVotes) &&
                Objects.equals(applicationDetails, that.applicationDetails) &&
                Objects.equals(loanMeta, that.loanMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, version, summary, description, trackerIssueKey, requestType, status, createdBy,
                updatedBy, createdAt, updatedAt, appliedAt, parameters, transferResponsible, transferVotes,
                applicationDetails, canUpdate, canCancel, canVote, canProvideOverCommitReserve, loanMeta,
                requestSubtype);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestDto{" +
                "id='" + id + '\'' +
                ", version='" + version + '\'' +
                ", summary='" + summary + '\'' +
                ", description='" + description + '\'' +
                ", trackerIssueKey='" + trackerIssueKey + '\'' +
                ", requestType=" + requestType +
                ", status=" + status +
                ", createdBy='" + createdBy + '\'' +
                ", updatedBy='" + updatedBy + '\'' +
                ", createdAt=" + createdAt +
                ", updatedAt=" + updatedAt +
                ", appliedAt=" + appliedAt +
                ", parameters=" + parameters +
                ", transferResponsible=" + transferResponsible +
                ", transferVotes=" + transferVotes +
                ", applicationDetails=" + applicationDetails +
                ", canUpdate=" + canUpdate +
                ", canCancel=" + canCancel +
                ", canVote=" + canVote +
                ", canProvideOverCommitReserve=" + canProvideOverCommitReserve +
                ", loanMeta=" + loanMeta +
                ", requestSubtype=" + requestSubtype +
                '}';
    }

    public static final class Builder {

        private String id;
        private String version;
        private String summary;
        private String description;
        private String trackerIssueKey;
        private TransferRequestTypeDto requestType;
        private TransferRequestSubtypeDto requestSubtype;
        private TransferRequestStatusDto status;
        private String createdBy;
        private String updatedBy;
        private Instant createdAt;
        private Instant updatedAt;
        private Instant appliedAt;
        private FrontTransferRequestParametersDto parameters;
        private FrontTransferRequestResponsibleDto transferResponsible;
        private FrontTransferRequestVotesDto transferVotes;
        private FrontTransferRequestApplicationDetailsDto applicationDetails;
        private Boolean canUpdate;
        private Boolean canCancel;
        private Boolean canVote;
        private Boolean canProvideOverCommitReserve;
        @Nullable
        private FrontTransferLoanMetaDto loanMeta;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder version(String version) {
            this.version = version;
            return this;
        }

        public Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder trackerIssueKey(String trackerIssueKey) {
            this.trackerIssueKey = trackerIssueKey;
            return this;
        }

        public Builder requestType(TransferRequestTypeDto requestType) {
            this.requestType = requestType;
            return this;
        }

        public Builder requestSubtype(TransferRequestSubtypeDto requestSubtype) {
            this.requestSubtype = requestSubtype;
            return this;
        }

        public Builder status(TransferRequestStatusDto status) {
            this.status = status;
            return this;
        }

        public Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public Builder updatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public Builder appliedAt(Instant appliedAt) {
            this.appliedAt = appliedAt;
            return this;
        }

        public Builder parameters(FrontTransferRequestParametersDto parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder transferResponsible(FrontTransferRequestResponsibleDto transferResponsible) {
            this.transferResponsible = transferResponsible;
            return this;
        }

        public Builder transferVotes(FrontTransferRequestVotesDto transferVotes) {
            this.transferVotes = transferVotes;
            return this;
        }

        public Builder applicationDetails(FrontTransferRequestApplicationDetailsDto applicationDetails) {
            this.applicationDetails = applicationDetails;
            return this;
        }

        public Builder canUpdate(boolean canUpdate) {
            this.canUpdate = canUpdate;
            return this;
        }

        public Builder canCancel(boolean canCancel) {
            this.canCancel = canCancel;
            return this;
        }

        public Builder canVote(boolean canVote) {
            this.canVote = canVote;
            return this;
        }

        public Builder canProvideOverCommitReserve(boolean canProvideOverCommitReserve) {
            this.canProvideOverCommitReserve = canProvideOverCommitReserve;
            return this;
        }

        public Builder loanMeta(@Nullable FrontTransferLoanMetaDto loanMeta) {
            this.loanMeta = loanMeta;
            return this;
        }

        public FrontTransferRequestDto build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(version, "Version is required");
            Preconditions.checkNotNull(requestType, "RequestType is required");
            Preconditions.checkNotNull(status, "Status is required");
            Preconditions.checkNotNull(createdBy, "CreatedBy is required");
            Preconditions.checkNotNull(createdAt, "CreatedAt is required");
            Preconditions.checkNotNull(parameters, "Parameters is required");
            Preconditions.checkNotNull(transferResponsible, "TransferResponsible is required");
            Preconditions.checkNotNull(transferVotes, "TransferVotes is required");


            Preconditions.checkNotNull(canUpdate, "CanUpdate is required");
            Preconditions.checkNotNull(canCancel, "CanCancel is required");
            Preconditions.checkNotNull(canVote, "CanVote is required");
            Preconditions.checkNotNull(canProvideOverCommitReserve, "CanProvideOverCommitReserve is required");
            return new FrontTransferRequestDto(id, version, summary, description, trackerIssueKey, requestType, status,
                    createdBy, updatedBy, createdAt, updatedAt, appliedAt, parameters, transferResponsible,
                    transferVotes, applicationDetails, canUpdate, canCancel, canVote, canProvideOverCommitReserve,
                    loanMeta, requestSubtype);
        }

    }

}
