package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;

/**
 * Transfer request field history DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request field history.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestFieldHistoryDto {

    private final String version;
    private final String summary;
    private final String description;
    private final String trackerIssueKey;
    private final TransferRequestTypeDto type;
    private final TransferRequestStatusDto status;

    @JsonCreator
    public FrontTransferRequestFieldHistoryDto(@JsonProperty("version") String version,
                                               @JsonProperty("summary") String summary,
                                               @JsonProperty("description") String description,
                                               @JsonProperty("trackerIssueKey") String trackerIssueKey,
                                               @JsonProperty("type") TransferRequestTypeDto type,
                                               @JsonProperty("status") TransferRequestStatusDto status) {
        this.version = version;
        this.summary = summary;
        this.description = description;
        this.trackerIssueKey = trackerIssueKey;
        this.type = type;
        this.status = status;
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request version.")
    public Optional<String> getVersion() {
        return Optional.ofNullable(version);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request summary.")
    public Optional<String> getSummary() {
        return Optional.ofNullable(summary);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request Tracker issue key.")
    public Optional<String> getTrackerIssueKey() {
        return Optional.ofNullable(trackerIssueKey);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request type.")
    public Optional<TransferRequestTypeDto> getType() {
        return Optional.ofNullable(type);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request type.")
    public Optional<TransferRequestStatusDto> getStatus() {
        return Optional.ofNullable(status);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestFieldHistoryDto that = (FrontTransferRequestFieldHistoryDto) o;
        return Objects.equals(version, that.version) &&
                Objects.equals(summary, that.summary) &&
                Objects.equals(description, that.description) &&
                Objects.equals(trackerIssueKey, that.trackerIssueKey) &&
                type == that.type &&
                status == that.status;
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, summary, description, trackerIssueKey, type, status);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestFieldHistoryDto{" +
                "version='" + version + '\'' +
                ", summary='" + summary + '\'' +
                ", description='" + description + '\'' +
                ", trackerIssueKey='" + trackerIssueKey + '\'' +
                ", type=" + type +
                ", status=" + status +
                '}';
    }

    public static final class Builder {

        private String version;
        private String summary;
        private String description;
        private String trackerIssueKey;
        private TransferRequestTypeDto type;
        private TransferRequestStatusDto status;

        private Builder() {
        }

        public Builder version(String version) {
            this.version = version;
            return this;
        }

        public Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder trackerIssueKey(String trackerIssueKey) {
            this.trackerIssueKey = trackerIssueKey;
            return this;
        }

        public Builder type(TransferRequestTypeDto type) {
            this.type = type;
            return this;
        }

        public Builder status(TransferRequestStatusDto status) {
            this.status = status;
            return this;
        }

        public FrontTransferRequestFieldHistoryDto build() {
            return new FrontTransferRequestFieldHistoryDto(version, summary, description, trackerIssueKey, type,
                    status);
        }

    }

}
