package ru.yandex.intranet.d.web.model.transfers.front;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestHistoryEventTypeDto;

/**
 * Transfer request history event DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request history event.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestHistoryEventDto {

    private final String id;
    private final TransferRequestHistoryEventTypeDto eventType;
    private final Instant eventTimestamp;
    private final String authorId;
    private final FrontTransferRequestFieldHistoryDto oldFields;
    private final FrontTransferRequestFieldHistoryDto newFields;
    private final FrontTransferRequestParametersDto oldParameters;
    private final FrontTransferRequestResponsibleDto oldTransferResponsible;
    private final FrontTransferRequestVotesDto oldTransferVotes;
    private final FrontTransferRequestApplicationDetailsDto oldApplicationDetails;
    private final FrontTransferRequestParametersDto newParameters;
    private final FrontTransferRequestResponsibleDto newTransferResponsible;
    private final FrontTransferRequestVotesDto newTransferVotes;
    private final FrontTransferRequestApplicationDetailsDto newApplicationDetails;
    private final @Nullable FrontTransferLoanMetaDto oldLoanMeta;
    private final @Nullable FrontTransferLoanMetaDto newLoanMeta;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public FrontTransferRequestHistoryEventDto(
            @JsonProperty("id") String id,
            @JsonProperty("eventType") TransferRequestHistoryEventTypeDto eventType,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            @JsonProperty("eventTimestamp") Instant eventTimestamp,
            @JsonProperty("authorId") String authorId,
            @JsonProperty("oldFields") FrontTransferRequestFieldHistoryDto oldFields,
            @JsonProperty("newFields") FrontTransferRequestFieldHistoryDto newFields,
            @JsonProperty("oldParameters") FrontTransferRequestParametersDto oldParameters,
            @JsonProperty("oldTransferResponsible") FrontTransferRequestResponsibleDto oldTransferResponsible,
            @JsonProperty("oldTransferVotes") FrontTransferRequestVotesDto oldTransferVotes,
            @JsonProperty("oldApplicationDetails") FrontTransferRequestApplicationDetailsDto oldApplicationDetails,
            @JsonProperty("newParameters") FrontTransferRequestParametersDto newParameters,
            @JsonProperty("newTransferResponsible") FrontTransferRequestResponsibleDto newTransferResponsible,
            @JsonProperty("newTransferVotes") FrontTransferRequestVotesDto newTransferVotes,
            @JsonProperty("newApplicationDetails") FrontTransferRequestApplicationDetailsDto newApplicationDetails,
            @JsonProperty("oldLoanMeta") @Nullable FrontTransferLoanMetaDto oldLoanMeta,
            @JsonProperty("newLoanMeta") @Nullable FrontTransferLoanMetaDto newLoanMeta) {
        this.id = id;
        this.eventType = eventType;
        this.eventTimestamp = eventTimestamp;
        this.authorId = authorId;
        this.oldFields = oldFields;
        this.newFields = newFields;
        this.oldParameters = oldParameters;
        this.oldTransferResponsible = oldTransferResponsible;
        this.oldTransferVotes = oldTransferVotes;
        this.oldApplicationDetails = oldApplicationDetails;
        this.newParameters = newParameters;
        this.newTransferResponsible = newTransferResponsible;
        this.newTransferVotes = newTransferVotes;
        this.newApplicationDetails = newApplicationDetails;
        this.oldLoanMeta = oldLoanMeta;
        this.newLoanMeta = newLoanMeta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request history event id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Event type.", required = true)
    public TransferRequestHistoryEventTypeDto getEventType() {
        return eventType;
    }

    @Schema(description = "Event timestamp.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getEventTimestamp() {
        return eventTimestamp;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Author id.")
    public Optional<String> getAuthorId() {
        return Optional.ofNullable(authorId);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old fields values.")
    public Optional<FrontTransferRequestFieldHistoryDto> getOldFields() {
        return Optional.ofNullable(oldFields);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New fields values.")
    public Optional<FrontTransferRequestFieldHistoryDto> getNewFields() {
        return Optional.ofNullable(newFields);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old request parameters.")
    public Optional<FrontTransferRequestParametersDto> getOldParameters() {
        return Optional.ofNullable(oldParameters);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old responsible.")
    public Optional<FrontTransferRequestResponsibleDto> getOldTransferResponsible() {
        return Optional.ofNullable(oldTransferResponsible);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old votes.")
    public Optional<FrontTransferRequestVotesDto> getOldTransferVotes() {
        return Optional.ofNullable(oldTransferVotes);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old application details.")
    public Optional<FrontTransferRequestApplicationDetailsDto> getOldApplicationDetails() {
        return Optional.ofNullable(oldApplicationDetails);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New request parameters.")
    public Optional<FrontTransferRequestParametersDto> getNewParameters() {
        return Optional.ofNullable(newParameters);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New responsible.")
    public Optional<FrontTransferRequestResponsibleDto> getNewTransferResponsible() {
        return Optional.ofNullable(newTransferResponsible);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New votes.")
    public Optional<FrontTransferRequestVotesDto> getNewTransferVotes() {
        return Optional.ofNullable(newTransferVotes);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New application details.")
    public Optional<FrontTransferRequestApplicationDetailsDto> getNewApplicationDetails() {
        return Optional.ofNullable(newApplicationDetails);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Old loan metadata.")
    public Optional<FrontTransferLoanMetaDto> getOldLoanMeta() {
        return Optional.ofNullable(oldLoanMeta);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "New loan metadata.")
    public Optional<FrontTransferLoanMetaDto> getNewLoanMeta() {
        return Optional.ofNullable(newLoanMeta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestHistoryEventDto that = (FrontTransferRequestHistoryEventDto) o;
        return Objects.equals(id, that.id) &&
                eventType == that.eventType &&
                Objects.equals(eventTimestamp, that.eventTimestamp) &&
                Objects.equals(authorId, that.authorId) &&
                Objects.equals(oldFields, that.oldFields) &&
                Objects.equals(newFields, that.newFields) &&
                Objects.equals(oldParameters, that.oldParameters) &&
                Objects.equals(oldTransferResponsible, that.oldTransferResponsible) &&
                Objects.equals(oldTransferVotes, that.oldTransferVotes) &&
                Objects.equals(oldApplicationDetails, that.oldApplicationDetails) &&
                Objects.equals(newParameters, that.newParameters) &&
                Objects.equals(newTransferResponsible, that.newTransferResponsible) &&
                Objects.equals(newTransferVotes, that.newTransferVotes) &&
                Objects.equals(newApplicationDetails, that.newApplicationDetails) &&
                Objects.equals(oldLoanMeta, that.oldLoanMeta) &&
                Objects.equals(newLoanMeta, that.newLoanMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, eventType, eventTimestamp, authorId, oldFields, newFields, oldParameters,
                oldTransferResponsible, oldTransferVotes, oldApplicationDetails, newParameters,
                newTransferResponsible, newTransferVotes, newApplicationDetails, oldLoanMeta, newLoanMeta);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestHistoryEventDto{" +
                "id='" + id + '\'' +
                ", eventType=" + eventType +
                ", eventTimestamp=" + eventTimestamp +
                ", authorId='" + authorId + '\'' +
                ", oldFields=" + oldFields +
                ", newFields=" + newFields +
                ", oldParameters=" + oldParameters +
                ", oldTransferResponsible=" + oldTransferResponsible +
                ", oldTransferVotes=" + oldTransferVotes +
                ", oldApplicationDetails=" + oldApplicationDetails +
                ", newParameters=" + newParameters +
                ", newTransferResponsible=" + newTransferResponsible +
                ", newTransferVotes=" + newTransferVotes +
                ", newApplicationDetails=" + newApplicationDetails +
                ", oldLoanMeta=" + oldLoanMeta +
                ", newLoanMeta=" + newLoanMeta +
                '}';
    }

    public static final class Builder {

        private String id;
        private TransferRequestHistoryEventTypeDto eventType;
        private Instant eventTimestamp;
        private String authorId;
        private FrontTransferRequestFieldHistoryDto oldFields;
        private FrontTransferRequestFieldHistoryDto newFields;
        private FrontTransferRequestParametersDto oldParameters;
        private FrontTransferRequestResponsibleDto oldTransferResponsible;
        private FrontTransferRequestVotesDto oldTransferVotes;
        private FrontTransferRequestApplicationDetailsDto oldApplicationDetails;
        private FrontTransferRequestParametersDto newParameters;
        private FrontTransferRequestResponsibleDto newTransferResponsible;
        private FrontTransferRequestVotesDto newTransferVotes;
        private FrontTransferRequestApplicationDetailsDto newApplicationDetails;
        private FrontTransferLoanMetaDto oldLoanMeta;
        private FrontTransferLoanMetaDto newLoanMeta;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder eventType(TransferRequestHistoryEventTypeDto eventType) {
            this.eventType = eventType;
            return this;
        }

        public Builder eventTimestamp(Instant eventTimestamp) {
            this.eventTimestamp = eventTimestamp;
            return this;
        }

        public Builder authorId(String authorId) {
            this.authorId = authorId;
            return this;
        }

        public Builder oldFields(FrontTransferRequestFieldHistoryDto oldFields) {
            this.oldFields = oldFields;
            return this;
        }

        public Builder newFields(FrontTransferRequestFieldHistoryDto newFields) {
            this.newFields = newFields;
            return this;
        }

        public Builder oldParameters(FrontTransferRequestParametersDto oldParameters) {
            this.oldParameters = oldParameters;
            return this;
        }

        public Builder oldTransferResponsible(FrontTransferRequestResponsibleDto oldTransferResponsible) {
            this.oldTransferResponsible = oldTransferResponsible;
            return this;
        }

        public Builder oldTransferVotes(FrontTransferRequestVotesDto oldTransferVotes) {
            this.oldTransferVotes = oldTransferVotes;
            return this;
        }

        public Builder oldApplicationDetails(FrontTransferRequestApplicationDetailsDto oldApplicationDetails) {
            this.oldApplicationDetails = oldApplicationDetails;
            return this;
        }

        public Builder newParameters(FrontTransferRequestParametersDto newParameters) {
            this.newParameters = newParameters;
            return this;
        }

        public Builder newTransferResponsible(FrontTransferRequestResponsibleDto newTransferResponsible) {
            this.newTransferResponsible = newTransferResponsible;
            return this;
        }

        public Builder newTransferVotes(FrontTransferRequestVotesDto newTransferVotes) {
            this.newTransferVotes = newTransferVotes;
            return this;
        }

        public Builder newApplicationDetails(FrontTransferRequestApplicationDetailsDto newApplicationDetails) {
            this.newApplicationDetails = newApplicationDetails;
            return this;
        }

        public Builder oldLoanMeta(FrontTransferLoanMetaDto oldLoanMeta) {
            this.oldLoanMeta = oldLoanMeta;
            return this;
        }

        public Builder newLoanMeta(FrontTransferLoanMetaDto newLoanMeta) {
            this.newLoanMeta = newLoanMeta;
            return this;
        }

        public FrontTransferRequestHistoryEventDto build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(eventType, "EventType is required");
            Preconditions.checkNotNull(eventTimestamp, "EventTimestamp is required");
            return new FrontTransferRequestHistoryEventDto(id, eventType, eventTimestamp, authorId, oldFields,
                    newFields, oldParameters, oldTransferResponsible, oldTransferVotes, oldApplicationDetails,
                    newParameters, newTransferResponsible, newTransferVotes, newApplicationDetails, oldLoanMeta,
                    newLoanMeta);
        }

    }

}
