package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request parameters.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestParametersDto {

    private final List<FrontQuotaTransferDto> quotaTransfers;
    private final List<FrontProvisionTransferDto> provisionTransfers;

    @JsonCreator
    public FrontTransferRequestParametersDto(
            @JsonProperty("quotaTransfers") List<FrontQuotaTransferDto> quotaTransfers,
            @JsonProperty("provisionTransfers") List<FrontProvisionTransferDto> provisionTransfers) {
        this.quotaTransfers = quotaTransfers != null ? quotaTransfers : List.of();
        this.provisionTransfers = provisionTransfers != null ? provisionTransfers : List.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Quota transfer requests.")
    public List<FrontQuotaTransferDto> getQuotaTransfers() {
        return quotaTransfers;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Provision transfer requests.")
    public List<FrontProvisionTransferDto> getProvisionTransfers() {
        return provisionTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestParametersDto that = (FrontTransferRequestParametersDto) o;
        return Objects.equals(quotaTransfers, that.quotaTransfers) &&
                Objects.equals(provisionTransfers, that.provisionTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(quotaTransfers, provisionTransfers);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestParametersDto{" +
                "quotaTransfers=" + quotaTransfers +
                ", provisionTransfers=" + provisionTransfers +
                '}';
    }

    public static final class Builder {

        private final List<FrontQuotaTransferDto> quotaTransfers = new ArrayList<>();
        private final List<FrontProvisionTransferDto> provisionTransfers = new ArrayList<>();

        private Builder() {
        }

        public Builder addQuotaTransfer(FrontQuotaTransferDto quotaTransfer) {
            this.quotaTransfers.add(quotaTransfer);
            return this;
        }

        public Builder addProvisionTransfer(FrontProvisionTransferDto provisionTransfer) {
            this.provisionTransfers.add(provisionTransfer);
            return this;
        }

        public FrontTransferRequestParametersDto build() {
            return new FrontTransferRequestParametersDto(quotaTransfers, provisionTransfers);
        }

    }

}
