package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request provider responsible DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request provider responsible.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestProviderResponsibleDto {

    private final String responsibleId;
    private final List<String> providerIds;

    @JsonCreator
    public FrontTransferRequestProviderResponsibleDto(@JsonProperty("responsibleId") String responsibleId,
                                                      @JsonProperty("providerIds") List<String> providerIds) {
        this.responsibleId = responsibleId;
        this.providerIds = providerIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "User id for provider responsible.", required = true)
    public String getResponsibleId() {
        return responsibleId;
    }

    @Schema(description = "Provider ids.", required = true)
    public List<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestProviderResponsibleDto that = (FrontTransferRequestProviderResponsibleDto) o;
        return Objects.equals(responsibleId, that.responsibleId) &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsibleId, providerIds);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestProviderResponsibleDto{" +
                "responsibleId='" + responsibleId + '\'' +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final List<String> providerIds = new ArrayList<>();

        private String responsibleId;

        private Builder() {
        }

        public Builder responsibleId(String responsibleId) {
            this.responsibleId = responsibleId;
            return this;
        }

        public Builder addProviderId(String providerId) {
            this.providerIds.add(providerId);
            return this;
        }

        public Builder addProviderIds(String... providerIds) {
            this.providerIds.addAll(Arrays.asList(providerIds));
            return this;
        }

        public Builder addProviderIds(Collection<? extends String> providerIds) {
            this.providerIds.addAll(providerIds);
            return this;
        }

        public FrontTransferRequestProviderResponsibleDto build() {
            Preconditions.checkNotNull(responsibleId, "ResponsibleId is required");
            return new FrontTransferRequestProviderResponsibleDto(responsibleId, providerIds);
        }

    }

}
