package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;

/**
 * Transfer request search DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request search parameters.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestSearchDto {

    private final String pageToken;
    private final Long limit;
    private final Boolean filterByCurrentUser;
    private final String filterByFolderId;
    private final String filterByServiceId;
    private final List<TransferRequestStatusDto> filterByStatus;

    @JsonCreator
    public FrontTransferRequestSearchDto(
            @JsonProperty("pageToken") String pageToken,
            @JsonProperty("limit") Long limit,
            @JsonProperty("filterByCurrentUser") Boolean filterByCurrentUser,
            @JsonProperty("filterByFolderId") String filterByFolderId,
            @JsonProperty("filterByServiceId") String filterByServiceId,
            @JsonProperty("filterByStatus") List<TransferRequestStatusDto> filterByStatus) {
        this.pageToken = pageToken;
        this.limit = limit;
        this.filterByCurrentUser = filterByCurrentUser;
        this.filterByFolderId = filterByFolderId;
        this.filterByServiceId = filterByServiceId;
        this.filterByStatus = filterByStatus;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Page token.")
    public Optional<String> getPageToken() {
        return Optional.ofNullable(pageToken);
    }

    @Schema(description = "Response items limit.")
    public Optional<Long> getLimit() {
        return Optional.ofNullable(limit);
    }

    @Schema(description = "'Find transfer requests where current user is responsible' flag.")
    public Optional<Boolean> getFilterByCurrentUser() {
        return Optional.ofNullable(filterByCurrentUser);
    }

    @Schema(description = "Filter transfer requests by folder id.")
    public Optional<String> getFilterByFolderId() {
        return Optional.ofNullable(filterByFolderId);
    }

    @Schema(description = "Vote transfer requests by service id.")
    public Optional<String> getFilterByServiceId() {
        return Optional.ofNullable(filterByServiceId);
    }

    @Schema(description = "Filter transfer requests by status.")
    public Optional<List<TransferRequestStatusDto>> getFilterByStatus() {
        return Optional.ofNullable(filterByStatus);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestSearchDto that = (FrontTransferRequestSearchDto) o;
        return Objects.equals(pageToken, that.pageToken) &&
                Objects.equals(limit, that.limit) &&
                Objects.equals(filterByCurrentUser, that.filterByCurrentUser) &&
                Objects.equals(filterByFolderId, that.filterByFolderId) &&
                Objects.equals(filterByServiceId, that.filterByServiceId) &&
                Objects.equals(filterByStatus, that.filterByStatus);
    }

    @Override
    public int hashCode() {
        return Objects.hash(pageToken, limit, filterByCurrentUser, filterByFolderId, filterByServiceId,
                filterByStatus);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestSearchDto{" +
                "pageToken='" + pageToken + '\'' +
                ", limit=" + limit +
                ", filterByCurrentUser=" + filterByCurrentUser +
                ", filterByFolderId='" + filterByFolderId + '\'' +
                ", filterByServiceId='" + filterByServiceId + '\'' +
                ", filterByStatus=" + filterByStatus +
                '}';
    }

    public static final class Builder {

        private final List<TransferRequestStatusDto> filterByStatus = new ArrayList<>();

        private String pageToken;
        private Long limit;
        private Boolean filterByCurrentUser;
        private String filterByFolderId;
        private String filterByServiceId;

        private Builder() {
        }

        public Builder pageToken(String pageToken) {
            this.pageToken = pageToken;
            return this;
        }

        public Builder limit(long limit) {
            this.limit = limit;
            return this;
        }

        public Builder filterByCurrentUser(boolean filterByCurrentUser) {
            this.filterByCurrentUser = filterByCurrentUser;
            return this;
        }

        public Builder filterByFolderId(String filterByFolderId) {
            this.filterByFolderId = filterByFolderId;
            return this;
        }

        public Builder filterByServiceId(String filterByServiceId) {
            this.filterByServiceId = filterByServiceId;
            return this;
        }

        public Builder addFilterByStatus(TransferRequestStatusDto transferRequestStatus) {
            this.filterByStatus.add(transferRequestStatus);
            return this;
        }

        public FrontTransferRequestSearchDto build() {
            return new FrontTransferRequestSearchDto(pageToken, limit, filterByCurrentUser, filterByFolderId,
                    filterByServiceId, filterByStatus);
        }

    }

}
