package ru.yandex.intranet.d.web.model.transfers.front;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestVoteTypeDto;

/**
 * Transfer request vote DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request vote.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestVoteDto {

    private final String userId;
    private final List<String> folderIds;
    private final List<String> serviceIds;
    private final Instant timestamp;
    private final TransferRequestVoteTypeDto voteType;
    private final List<String> providerIds;

    @JsonCreator
    public FrontTransferRequestVoteDto(
            @JsonProperty("userId") String userId,
            @JsonProperty("folderIds") List<String> folderIds,
            @JsonProperty("serviceIds") List<String> serviceIds,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            @JsonProperty("timestamp") Instant timestamp,
            @JsonProperty("voteType") TransferRequestVoteTypeDto voteType,
            @JsonProperty("providerIds") List<String> providerIds) {
        this.userId = userId;
        this.folderIds = folderIds;
        this.serviceIds = serviceIds;
        this.timestamp = timestamp;
        this.voteType = voteType;
        this.providerIds = providerIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "User id.", required = true)
    public String getUserId() {
        return userId;
    }

    @Schema(description = "Folder ids.", required = true)
    public List<String> getFolderIds() {
        return folderIds;
    }

    @Schema(description = "Service ids.", required = true)
    public List<String> getServiceIds() {
        return serviceIds;
    }

    @Schema(description = "Vote timestamp.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getTimestamp() {
        return timestamp;
    }

    @Schema(description = "Vote type.", required = true)
    public TransferRequestVoteTypeDto getVoteType() {
        return voteType;
    }

    @Schema(description = "Provider ids when user is responsible for those providers.", required = true)
    public List<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestVoteDto that = (FrontTransferRequestVoteDto) o;
        return Objects.equals(userId, that.userId) &&
                Objects.equals(folderIds, that.folderIds) &&
                Objects.equals(serviceIds, that.serviceIds) &&
                Objects.equals(timestamp, that.timestamp) &&
                voteType == that.voteType &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userId, folderIds, serviceIds, timestamp, voteType, providerIds);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestVoteDto{" +
                "userId='" + userId + '\'' +
                ", folderIds=" + folderIds +
                ", serviceIds=" + serviceIds +
                ", timestamp=" + timestamp +
                ", voteType=" + voteType +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final List<String> folderIds = new ArrayList<>();
        private final List<String> serviceIds = new ArrayList<>();
        private final List<String> providerIds = new ArrayList<>();

        private String userId;
        private Instant timestamp;
        private TransferRequestVoteTypeDto voteType;

        private Builder() {
        }

        public Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        public Builder addFolderId(String folderId) {
            this.folderIds.add(folderId);
            return this;
        }

        public Builder addServiceId(String serviceId) {
            this.serviceIds.add(serviceId);
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Builder voteType(TransferRequestVoteTypeDto voteType) {
            this.voteType = voteType;
            return this;
        }

        public Builder addProviderId(String providerId) {
            this.providerIds.add(providerId);
            return this;
        }

        public FrontTransferRequestVoteDto build() {
            Preconditions.checkNotNull(userId, "UserId is required");
            Preconditions.checkNotNull(timestamp, "Timestamp is required");
            Preconditions.checkNotNull(voteType, "VoteType is required");
            return new FrontTransferRequestVoteDto(userId, folderIds, serviceIds, timestamp, voteType, providerIds);
        }

    }

}
