package ru.yandex.intranet.d.web.model.transfers.front;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountsSpaceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontFolderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontProviderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentationDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceTypeDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontServiceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUnitDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUserDictionaryElementDto;

/**
 * Single transfer requests page DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer requests page.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontTransferRequestsPageDto {

    private final List<FrontTransferRequestDto> transfers;
    private final Map<String, FrontFolderDictionaryElementDto> folders;
    private final Map<String, FrontServiceDictionaryElementDto> services;
    private final Map<String, FrontResourceDictionaryElementDto> resources;
    private final Map<String, FrontProviderDictionaryElementDto> providers;
    private final Map<String, FrontUnitDictionaryElementDto> units;
    private final Map<String, FrontUserDictionaryElementDto> users;
    private final String nextPageToken;
    private final Map<String, FrontAccountDictionaryElementDto> accounts;
    private final Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes;
    private final Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces;
    private final Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations;
    private final Map<String, FrontResourceSegmentDictionaryElementDto> segments;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public FrontTransferRequestsPageDto(
            List<FrontTransferRequestDto> transfers,
            Map<String, FrontFolderDictionaryElementDto> folders,
            Map<String, FrontServiceDictionaryElementDto> services,
            Map<String, FrontResourceDictionaryElementDto> resources,
            Map<String, FrontProviderDictionaryElementDto> providers,
            Map<String, FrontUnitDictionaryElementDto> units,
            Map<String, FrontUserDictionaryElementDto> users,
            String nextPageToken,
            Map<String, FrontAccountDictionaryElementDto> accounts,
            Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes,
            Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces,
            Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations,
            Map<String, FrontResourceSegmentDictionaryElementDto> segments
    ) {
        this.transfers = transfers;
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.providers = providers;
        this.units = units;
        this.users = users;
        this.nextPageToken = nextPageToken;
        this.accounts = accounts;
        this.resourceTypes = resourceTypes;
        this.accountsSpaces = accountsSpaces;
        this.segmentations = segmentations;
        this.segments = segments;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer requests.", required = true)
    public List<FrontTransferRequestDto> getTransfers() {
        return transfers;
    }

    @Schema(description = "Folders by id.", required = true)
    public Map<String, FrontFolderDictionaryElementDto> getFolders() {
        return folders;
    }

    @Schema(description = "Services by id.", required = true)
    public Map<String, FrontServiceDictionaryElementDto> getServices() {
        return services;
    }

    @Schema(description = "Resources by id.", required = true)
    public Map<String, FrontResourceDictionaryElementDto> getResources() {
        return resources;
    }

    @Schema(description = "Providers by id.", required = true)
    public Map<String, FrontProviderDictionaryElementDto> getProviders() {
        return providers;
    }

    @Schema(description = "Units of measure by id.", required = true)
    public Map<String, FrontUnitDictionaryElementDto> getUnits() {
        return units;
    }

    @Schema(description = "Users by id.", required = true)
    public Map<String, FrontUserDictionaryElementDto> getUsers() {
        return users;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Next page token.")
    public Optional<String> getNextPageToken() {
        return Optional.ofNullable(nextPageToken);
    }

    @Schema(description = "Accounts by id.", required = true)
    public Map<String, FrontAccountDictionaryElementDto> getAccounts() {
        return accounts;
    }

    @Schema(description = "Resource types by id.", required = true)
    public Map<String, FrontResourceTypeDictionaryElementDto> getResourceTypes() {
        return resourceTypes;
    }

    @Schema(description = "Accounts spaces by id.", required = true)
    public Map<String, FrontAccountsSpaceDictionaryElementDto> getAccountsSpaces() {
        return accountsSpaces;
    }

    @Schema(description = "Resource segmentations by id.", required = true)
    public Map<String, FrontResourceSegmentationDictionaryElementDto> getSegmentations() {
        return segmentations;
    }

    @Schema(description = "Resource segments by id.", required = true)
    public Map<String, FrontResourceSegmentDictionaryElementDto> getSegments() {
        return segments;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontTransferRequestsPageDto that = (FrontTransferRequestsPageDto) o;
        return Objects.equals(transfers, that.transfers) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(units, that.units) &&
                Objects.equals(users, that.users) &&
                Objects.equals(nextPageToken, that.nextPageToken) &&
                Objects.equals(accounts, that.accounts) &&
                Objects.equals(resourceTypes, that.resourceTypes) &&
                Objects.equals(accountsSpaces, that.accountsSpaces) &&
                Objects.equals(segmentations, that.segmentations) &&
                Objects.equals(segments, that.segments);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transfers, folders, services, resources, providers, units, users, nextPageToken, accounts,
                resourceTypes, accountsSpaces, segmentations, segments);
    }

    @Override
    public String toString() {
        return "FrontTransferRequestsPageDto{" +
                "transfers=" + transfers +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", providers=" + providers +
                ", units=" + units +
                ", users=" + users +
                ", nextPageToken='" + nextPageToken + '\'' +
                ", accounts=" + accounts +
                ", resourceTypes=" + resourceTypes +
                ", accountsSpaces=" + accountsSpaces +
                ", segmentations=" + segmentations +
                ", segments=" + segments +
                '}';
    }

    public static final class Builder implements DictionaryBuilder {

        private final List<FrontTransferRequestDto> transfers = new ArrayList<>();
        private final Map<String, FrontFolderDictionaryElementDto> folders = new HashMap<>();
        private final Map<String, FrontServiceDictionaryElementDto> services = new HashMap<>();
        private final Map<String, FrontResourceDictionaryElementDto> resources = new HashMap<>();
        private final Map<String, FrontProviderDictionaryElementDto> providers = new HashMap<>();
        private final Map<String, FrontUnitDictionaryElementDto> units = new HashMap<>();
        private final Map<String, FrontUserDictionaryElementDto> users = new HashMap<>();
        private final Map<String, FrontAccountDictionaryElementDto> accounts = new HashMap<>();
        private final Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes = new HashMap<>();
        private final Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces = new HashMap<>();
        private final Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations = new HashMap<>();
        private final Map<String, FrontResourceSegmentDictionaryElementDto> segments = new HashMap<>();

        private String nextPageToken;

        private Builder() {
        }

        public Builder addTransfer(FrontTransferRequestDto transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public Builder addFolder(String folderId, FrontFolderDictionaryElementDto folder) {
            this.folders.put(folderId, folder);
            return this;
        }

        public Builder addService(String serviceId, FrontServiceDictionaryElementDto service) {
            this.services.put(serviceId, service);
            return this;
        }

        public Builder addResource(String resourceId, FrontResourceDictionaryElementDto resource) {
            this.resources.put(resourceId, resource);
            return this;
        }

        public Builder addProvider(String providerId, FrontProviderDictionaryElementDto provider) {
            this.providers.put(providerId, provider);
            return this;
        }

        public Builder addUnit(String unitId, FrontUnitDictionaryElementDto unit) {
            this.units.put(unitId, unit);
            return this;
        }

        public Builder addUser(String userId, FrontUserDictionaryElementDto user) {
            this.users.put(userId, user);
            return this;
        }

        @Override
        public DictionaryBuilder addAccount(String accountId, FrontAccountDictionaryElementDto account) {
            accounts.put(accountId, account);
            return this;
        }

        public Builder nextPageToken(String nextPageToken) {
            this.nextPageToken = nextPageToken;
            return this;
        }

        @Override
        public Builder addResourceType(String resourceTypeId, FrontResourceTypeDictionaryElementDto resourceType) {
            this.resourceTypes.put(resourceTypeId, resourceType);
            return this;
        }

        @Override
        public Builder addAccountsSpace(String accountsSpaceId, FrontAccountsSpaceDictionaryElementDto accountsSpace) {
            this.accountsSpaces.put(accountsSpaceId, accountsSpace);
            return this;
        }

        @Override
        public Builder addSegmentation(
                String segmentationId, FrontResourceSegmentationDictionaryElementDto segmentation
        ) {
            segmentations.put(segmentationId, segmentation);
            return this;
        }

        @Override
        public Builder addSegment(String segmentId, FrontResourceSegmentDictionaryElementDto segment) {
            segments.put(segmentId, segment);
            return this;
        }

        public List<FrontTransferRequestDto> getTransfers() {
            return transfers;
        }

        public FrontTransferRequestsPageDto build() {
            return new FrontTransferRequestsPageDto(transfers, folders, services, resources, providers, units, users,
                    nextPageToken, accounts, resourceTypes, accountsSpaces, segmentations, segments);
        }

    }

}
