package ru.yandex.intranet.d.web.model.transfers.front.dryrun;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.ErrorCollectionDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountsSpaceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontFolderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontProviderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentationDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceTypeDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontServiceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUnitDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUserDictionaryElementDto;
import ru.yandex.intranet.d.web.model.transfers.front.DictionaryBuilder;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestResponsibleDto;

/**
 * FrontDryRunSingleTransferRequestDto
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */

@Schema(description = "Single dry-run transfer result.")
public final class FrontDryRunSingleTransferResultDto {

    private final Map<String, FrontFolderDictionaryElementDto> folders;
    private final Map<String, FrontServiceDictionaryElementDto> services;
    private final Map<String, FrontResourceDictionaryElementDto> resources;
    private final Map<String, FrontProviderDictionaryElementDto> providers;
    private final Map<String, FrontUnitDictionaryElementDto> units;
    private final Map<String, FrontUserDictionaryElementDto> users;
    private final Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes;
    private final Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces;
    private final Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations;
    private final Map<String, FrontResourceSegmentDictionaryElementDto> segments;

    private final FrontDryRunTransferRequestDto transferRequestDto;
    private final FrontDryRunTransferPermissionsDto permissions;
    private final FrontTransferRequestResponsibleDto transferResponsible;
    private final FrontDryRunTransferWarningsDto warnings;
    private final Map<String, FrontAccountDictionaryElementDto> accounts;
    private final ErrorCollectionDto errors;


    @SuppressWarnings("ParameterNumber")
    public FrontDryRunSingleTransferResultDto(
            FrontDryRunTransferRequestDto transfer,
            FrontDryRunTransferPermissionsDto permissions,
            FrontTransferRequestResponsibleDto transferResponsible,
            Map<String, FrontFolderDictionaryElementDto> folders,
            Map<String, FrontServiceDictionaryElementDto> services,
            Map<String, FrontResourceDictionaryElementDto> resources,
            Map<String, FrontProviderDictionaryElementDto> providers,
            Map<String, FrontUnitDictionaryElementDto> units,
            Map<String, FrontUserDictionaryElementDto> users,
            Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes,
            Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces,
            Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations,
            Map<String, FrontResourceSegmentDictionaryElementDto> segments,
            FrontDryRunTransferWarningsDto warnings,
            Map<String, FrontAccountDictionaryElementDto> accounts,
            ErrorCollectionDto errors
    ) {
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.providers = providers;
        this.units = units;
        this.users = users;
        this.transferRequestDto = transfer;
        this.permissions = permissions;
        this.transferResponsible = transferResponsible;
        this.resourceTypes = resourceTypes;
        this.accountsSpaces = accountsSpaces;
        this.segmentations = segmentations;
        this.segments = segments;
        this.warnings = warnings;
        this.accounts = accounts;
        this.errors = errors;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Folders by id.", required = true)
    public Map<String, FrontFolderDictionaryElementDto> getFolders() {
        return folders;
    }

    @Schema(description = "Services by id.", required = true)
    public Map<String, FrontServiceDictionaryElementDto> getServices() {
        return services;
    }

    @Schema(description = "Resources by id.", required = true)
    public Map<String, FrontResourceDictionaryElementDto> getResources() {
        return resources;
    }

    @Schema(description = "Providers by id.", required = true)
    public Map<String, FrontProviderDictionaryElementDto> getProviders() {
        return providers;
    }

    @Schema(description = "Units of measure by id.", required = true)
    public Map<String, FrontUnitDictionaryElementDto> getUnits() {
        return units;
    }

    @Schema(description = "Users by id.", required = true)
    public Map<String, FrontUserDictionaryElementDto> getUsers() {
        return users;
    }

    @Schema(description = "Transfer request.", required = true)
    public FrontDryRunTransferRequestDto getTransfer() {
        return transferRequestDto;
    }

    @Schema(description = "Transfer request permissions.")
    public Optional<FrontDryRunTransferPermissionsDto> getPermissions() {
        return Optional.ofNullable(permissions);
    }

    @Schema(description = "Transfer request responsible.")
    public Optional<FrontTransferRequestResponsibleDto> getTransferResponsible() {
        return Optional.ofNullable(transferResponsible);
    }

    @Schema(description = "Transfer dry-run request warnings.", required = true)
    public FrontDryRunTransferWarningsDto getWarnings() {
        return warnings;
    }

    @Schema(description = "Accounts by id.", required = true)
    public Map<String, FrontAccountDictionaryElementDto> getAccounts() {
        return accounts;
    }

    @Schema(description = "Quotas validation errors.", required = true)
    public ErrorCollectionDto getErrors() {
        return errors;
    }

    @Schema(description = "Resource types by id.", required = true)
    public Map<String, FrontResourceTypeDictionaryElementDto> getResourceTypes() {
        return resourceTypes;
    }

    @Schema(description = "Accounts spaces by id.", required = true)
    public Map<String, FrontAccountsSpaceDictionaryElementDto> getAccountsSpaces() {
        return accountsSpaces;
    }

    @Schema(description = "Resource segmentations by id.", required = true)
    public Map<String, FrontResourceSegmentationDictionaryElementDto> getSegmentations() {
        return segmentations;
    }

    @Schema(description = "Resource segments by id.", required = true)
    public Map<String, FrontResourceSegmentDictionaryElementDto> getSegments() {
        return segments;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontDryRunSingleTransferResultDto that = (FrontDryRunSingleTransferResultDto) o;
        return Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(units, that.units) &&
                Objects.equals(users, that.users) &&
                Objects.equals(resourceTypes, that.resourceTypes) &&
                Objects.equals(accountsSpaces, that.accountsSpaces) &&
                Objects.equals(segmentations, that.segmentations) &&
                Objects.equals(segments, that.segments) &&
                Objects.equals(transferRequestDto, that.transferRequestDto) &&
                Objects.equals(permissions, that.permissions) &&
                Objects.equals(transferResponsible, that.transferResponsible) &&
                Objects.equals(warnings, that.warnings) &&
                Objects.equals(accounts, that.accounts) &&
                Objects.equals(errors, that.errors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                folders, services, resources, providers, units, users, resourceTypes, accountsSpaces, segmentations,
                segments, transferRequestDto, permissions, transferResponsible, warnings, accounts, errors
        );
    }

    @Override
    public String toString() {
        return "FrontDryRunSingleTransferResultDto{" +
                "folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", providers=" + providers +
                ", units=" + units +
                ", users=" + users +
                ", resourceTypes=" + resourceTypes +
                ", accountsSpaces=" + accountsSpaces +
                ", segmentations=" + segmentations +
                ", segments=" + segments +
                ", transferRequestDto=" + transferRequestDto +
                ", permissions=" + permissions +
                ", transferResponsible=" + transferResponsible +
                ", warnings=" + warnings +
                ", accounts=" + accounts +
                ", errors=" + errors +
                '}';
    }

    public static class Builder implements DictionaryBuilder {
        private FrontDryRunTransferRequestDto transferRequestDto;
        private FrontDryRunTransferPermissionsDto permissions;
        private FrontTransferRequestResponsibleDto transferResponsible;
        private FrontDryRunTransferWarningsDto warnings;
        private ErrorCollectionDto errors;

        private final Map<String, FrontFolderDictionaryElementDto> folders = new HashMap<>();
        private final Map<String, FrontServiceDictionaryElementDto> services = new HashMap<>();
        private final Map<String, FrontResourceDictionaryElementDto> resources = new HashMap<>();
        private final Map<String, FrontProviderDictionaryElementDto> providers = new HashMap<>();
        private final Map<String, FrontUnitDictionaryElementDto> units = new HashMap<>();
        private final Map<String, FrontUserDictionaryElementDto> users = new HashMap<>();
        private final Map<String, FrontAccountDictionaryElementDto> accounts = new HashMap<>();
        private final Map<String, FrontResourceTypeDictionaryElementDto> resourceTypes = new HashMap<>();
        private final Map<String, FrontAccountsSpaceDictionaryElementDto> accountsSpaces = new HashMap<>();
        private final Map<String, FrontResourceSegmentationDictionaryElementDto> segmentations = new HashMap<>();
        private final Map<String, FrontResourceSegmentDictionaryElementDto> segments = new HashMap<>();

        public Builder transfer(FrontDryRunTransferRequestDto transfer) {
            this.transferRequestDto = transfer;
            return this;
        }

        public Builder permissions(FrontDryRunTransferPermissionsDto permissions) {
            this.permissions = permissions;
            return this;
        }

        public Builder responsible(FrontTransferRequestResponsibleDto responsible) {
            this.transferResponsible = responsible;
            return this;
        }

        public Builder warnings(FrontDryRunTransferWarningsDto warnings) {
            this.warnings = warnings;
            return this;
        }

        public Builder addFolder(String id, FrontFolderDictionaryElementDto folderDictionary) {
            folders.put(id, folderDictionary);
            return this;
        }

        public Builder addService(String id, FrontServiceDictionaryElementDto serviceDictionary) {
            services.put(id, serviceDictionary);
            return this;
        }

        public Builder addResource(String id, FrontResourceDictionaryElementDto resourceDictionary) {
            resources.put(id, resourceDictionary);
            return this;
        }

        public Builder addProvider(String id, FrontProviderDictionaryElementDto providerDictionary) {
            providers.put(id, providerDictionary);
            return this;
        }

        public Builder addUnit(String id, FrontUnitDictionaryElementDto unitDictionary) {
            units.put(id, unitDictionary);
            return this;
        }

        public Builder addUser(String id, FrontUserDictionaryElementDto userDictionary) {
            users.put(id, userDictionary);
            return this;
        }

        public Builder errors(ErrorCollectionDto errors) {
            this.errors = errors;
            return this;
        }

        @Override
        public DictionaryBuilder addAccount(String accountId, FrontAccountDictionaryElementDto account) {
            accounts.put(accountId, account);
            return this;
        }

        @Override
        public Builder addResourceType(String resourceTypeId, FrontResourceTypeDictionaryElementDto resourceType) {
            this.resourceTypes.put(resourceTypeId, resourceType);
            return this;
        }

        @Override
        public Builder addAccountsSpace(String accountsSpaceId, FrontAccountsSpaceDictionaryElementDto accountsSpace) {
            this.accountsSpaces.put(accountsSpaceId, accountsSpace);
            return this;
        }

        @Override
        public Builder addSegmentation(
                String segmentationId, FrontResourceSegmentationDictionaryElementDto segmentation
        ) {
            segmentations.put(segmentationId, segmentation);
            return this;
        }

        @Override
        public Builder addSegment(String segmentId, FrontResourceSegmentDictionaryElementDto segment) {
            segments.put(segmentId, segment);
            return this;
        }


        public FrontDryRunSingleTransferResultDto build() {
            Preconditions.checkNotNull(transferRequestDto, "Transfer is required");
            Preconditions.checkNotNull(warnings, "Warnings is required");
            Preconditions.checkNotNull(errors, "Errors is required");
            return new FrontDryRunSingleTransferResultDto(
                    transferRequestDto,
                    permissions,
                    transferResponsible,
                    folders,
                    services,
                    resources,
                    providers,
                    units,
                    users,
                    resourceTypes, accountsSpaces, segmentations, segments, warnings,
                    accounts,
                    errors);
        }
    }
}
