package ru.yandex.intranet.d.web.model.units.front;

import java.util.Locale;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;

/**
 * FrontUnitDto
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Schema(description = "Unit of measurement.")
@JsonIgnoreProperties(ignoreUnknown = true)
public class FrontUnitDto {

    private final String id;
    private final String key;
    private final Map<PluralFormDto, String> shortName;
    private final Map<PluralFormDto, String> longName;
    private final long base;
    private final long power;

    @SuppressWarnings("ParameterNumber")
    @JsonCreator
    public FrontUnitDto(String id,
                        String key,
                        Map<PluralFormDto, String> shortName,
                        Map<PluralFormDto, String> longName,
                        long base,
                        long power) {
        this.id = id;
        this.key = key;
        this.shortName = shortName;
        this.longName = longName;
        this.base = base;
        this.power = power;
    }

    public FrontUnitDto(UnitModel model, Locale locale) {
        String longMany = Locales.select(model.getLongNamePluralEn(),
                model.getLongNamePluralRu().get(GrammaticalCase.GENITIVE), locale);
        Map<PluralFormDto, String> longNames = Map.of(
                PluralFormDto.ONE,
                Locales.select(model.getLongNameSingularEn(), model.getLongNameSingularRu()
                        .get(GrammaticalCase.NOMINATIVE), locale),
                PluralFormDto.SOME,
                Locales.select(model.getLongNamePluralEn(), model.getLongNameSingularRu()
                        .get(GrammaticalCase.GENITIVE), locale),
                PluralFormDto.NONE, longMany,
                PluralFormDto.MANY, longMany
        );
        String shortMany = Locales.select(model.getShortNamePluralEn(),
                model.getShortNamePluralRu().get(GrammaticalCase.GENITIVE), locale);
        Map<PluralFormDto, String> shortNames = Map.of(
                PluralFormDto.ONE,
                Locales.select(model.getShortNameSingularEn(), model.getShortNameSingularRu()
                        .get(GrammaticalCase.NOMINATIVE), locale),
                PluralFormDto.SOME,
                Locales.select(model.getShortNamePluralEn(), model.getShortNameSingularRu()
                        .get(GrammaticalCase.GENITIVE), locale),
                PluralFormDto.NONE, shortMany,
                PluralFormDto.MANY, shortMany
        );
        this.id = model.getId();
        this.key = model.getKey();
        this.shortName = shortNames;
        this.longName = longNames;
        this.base = model.getBase();
        this.power = model.getPower();
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Key, unique per ensemble.", required = true)
    public String getKey() {
        return key;
    }

    @Schema(description = "Short name.", required = true)
    public Map<PluralFormDto, String> getShortName() {
        return shortName;
    }

    @Schema(description = "Long name.", required = true)
    public Map<PluralFormDto, String> getLongName() {
        return longName;
    }

    @Schema(description = "Unit multiplier base.", required = true)
    public long getBase() {
        return base;
    }

    @Schema(description = "Unit multiplier power.", required = true)
    public long getPower() {
        return power;
    }
}
