package ru.yandex.intranet.d.web.model.units.front;

import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;

/**
 * FrontUnitsEnsembleDto
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */

@Schema(description = "Units of measurement ensemble.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontUnitsEnsembleDto {

    private final String id;
    private final long version;
    private final String name;
    private final String description;
    private final boolean fractionsAllowed;
    private final List<FrontUnitDto> units;
    private final String key;

    @SuppressWarnings("ParameterNumber")
    @JsonCreator
    public FrontUnitsEnsembleDto(String id,
                                 long version,
                                 String name,
                                 String description,
                                 boolean fractionsAllowed,
                                 List<FrontUnitDto> units,
                                 String key) {
        this.id = id;
        this.version = version;
        this.name = name;
        this.description = description;
        this.fractionsAllowed = fractionsAllowed;
        this.units = units;
        this.key = key;
    }

    public FrontUnitsEnsembleDto(UnitsEnsembleModel model, Locale locale) {
        this.id = model.getId();
        this.version = model.getVersion();
        this.name = Locales.select(model.getNameEn(), model.getNameRu(), locale);
        this.description = Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale);
        this.fractionsAllowed = model.isFractionsAllowed();
        this.units = model.getUnits().stream().sorted(UnitsComparator.INSTANCE)
                .map(u -> new FrontUnitDto(u, locale)).collect(Collectors.toList());
        this.key = model.getKey();
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Ensemble version.", required = true)
    public long getVersion() {
        return version;
    }

    @Schema(description = "Name.", required = true)
    public String getName() {
        return name;
    }

    @Schema(description = "Description.", required = true)
    public String getDescription() {
        return description;
    }

    @Schema(description = "Fractions allowed flag.", required = true)
    public boolean isFractionsAllowed() {
        return fractionsAllowed;
    }

    @Schema(description = "Units of measurement.", required = true)
    public List<FrontUnitDto> getUnits() {
        return units;
    }

    @Schema(description = "Unique key.", required = true)
    public String getKey() {
        return key;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontUnitsEnsembleDto that = (FrontUnitsEnsembleDto) o;
        return version == that.version &&
                fractionsAllowed == that.fractionsAllowed &&
                Objects.equals(id, that.id) &&
                Objects.equals(name, that.name) &&
                Objects.equals(description, that.description) &&
                Objects.equals(units, that.units) &&
                Objects.equals(key, that.key);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, version, name, description, fractionsAllowed, units, key);
    }

    @Override
    public String toString() {
        return "UnitsEnsembleDto{" +
                "id='" + id + '\'' +
                ", version=" + version +
                ", name='" + name + '\'' +
                ", description='" + description + '\'' +
                ", fractionsAllowed=" + fractionsAllowed +
                ", units=" + units +
                ", key='" + key + '\'' +
                '}';
    }
}
