package ru.yandex.intranet.d.web.security.blackbox;

import java.net.URI;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Splitter;
import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.MimeType;
import org.springframework.web.reactive.function.BodyExtractors;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions;
import org.springframework.web.reactive.function.client.UnknownHttpStatusCodeException;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;

import ru.yandex.intranet.d.util.ResolverHolder;
import ru.yandex.intranet.d.util.http.YaReactorClientHttpConnector;
import ru.yandex.intranet.d.web.security.blackbox.model.BlackboxException;
import ru.yandex.intranet.d.web.security.blackbox.model.BlackboxOAuthResponse;
import ru.yandex.intranet.d.web.security.blackbox.model.BlackboxSessionIdResponse;
import ru.yandex.intranet.d.web.security.blackbox.model.CheckedOAuthToken;
import ru.yandex.intranet.d.web.security.blackbox.model.CheckedSessionId;
import ru.yandex.intranet.d.web.security.blackbox.model.InvalidOAuthToken;
import ru.yandex.intranet.d.web.security.blackbox.model.InvalidSessionId;
import ru.yandex.intranet.d.web.security.blackbox.model.ValidOAuthToken;
import ru.yandex.intranet.d.web.security.blackbox.model.ValidSessionId;

/**
 * Blackbox client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
@Profile({"dev", "testing", "production"})
public class BlackboxClient {

    private static final long RESPONSE_SIZE_LIMIT = 10485760L;
    private static final byte[] EMPTY = new byte[0];

    private final String baseUrl;
    private final String userAgent;
    private final WebClient client;

    public BlackboxClient(@Value("${blackbox.client.connectTimeoutMs}") int connectTimeoutMillis,
                          @Value("${blackbox.client.readTimeoutMs}") long readTimeoutMillis,
                          @Value("${blackbox.client.writeTimeoutMs}") long writeTimeoutMillis,
                          @Value("${blackbox.client.baseUrl}") String baseUrl,
                          @Value("${http.client.userAgent}") String userAgent) {
        this.baseUrl = baseUrl;
        this.userAgent = userAgent;
        this.client = createWebClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis);
    }

    public Mono<CheckedSessionId> sessionId(String serviceTicket, String sessionId, String userIp, String host) {
        return sessionId(serviceTicket, sessionId, userIp, host, null);
    }

    public Mono<CheckedSessionId> sessionId(String serviceTicket, String sessionId, String userIp, String host,
                                            String sslSessionId) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .queryParam("method", "sessionid")
                .queryParam("sessionid", sessionId)
                .queryParam("userip", userIp)
                .queryParam("host", host)
                .queryParam("format", "json");
        if (sslSessionId != null) {
            builder.queryParam("sslsessionid", sslSessionId);
        }
        final String uri = builder.toUriString();
        return client.get()
                .uri(uri)
                .header("X-Ya-Service-Ticket", serviceTicket)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(BlackboxSessionIdResponse.class).flatMap(this::fromSessionIdResponse);
                    } else {
                        return createException(r, uri).flatMap(Mono::error);
                    }
                });
    }

    public Mono<CheckedOAuthToken> oauth(String serviceTicket, String token, String userIp) {
        return oauth(serviceTicket, token, userIp, Collections.emptyList());
    }

    public Mono<CheckedOAuthToken> oauth(String serviceTicket, String token, String userIp, List<String> scopes) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .queryParam("method", "oauth")
                .queryParam("userip", userIp)
                .queryParam("format", "json");
        if (!scopes.isEmpty()) {
            builder.queryParam("scopes", String.join(",", scopes));
        }
        final String uri = builder.toUriString();
        return client.get()
                .uri(uri)
                .header("X-Ya-Service-Ticket", serviceTicket)
                .header("Authorization", "OAuth " + token)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(BlackboxOAuthResponse.class).flatMap(this::fromOAuthResponse);
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    private Mono<CheckedSessionId> fromSessionIdResponse(BlackboxSessionIdResponse resp) {
        if (resp.getException().isPresent()
                && resp.getException().get().getId() != BlackboxSessionIdResponse.BlackboxException.OK) {
            return Mono.error(new BlackboxException(resp.getException().get().getId(),
                    resp.getException().get().getValue(), resp.getError().orElse("")));
        }
        if (resp.getStatus().isEmpty()) {
            return Mono.error(new IllegalStateException("Missing status in blackbox response"));
        }
        if (resp.getStatus().get().getId() != BlackboxSessionIdResponse.Status.VALID
                && resp.getStatus().get().getId() != BlackboxSessionIdResponse.Status.NEED_RESET) {
            String error = resp.getError().orElse("");
            InvalidSessionId.Status status = new InvalidSessionId.Status(resp.getStatus().get().getId(),
                    resp.getStatus().get().getValue());
            InvalidSessionId invalid = new InvalidSessionId(error, status);
            return Mono.just(CheckedSessionId.invalid(invalid));
        }
        if (resp.getUid().isEmpty()) {
            return Mono.error(new IllegalStateException("Missing uid in blackbox response"));
        }
        boolean needReset = resp.getStatus().get().getId() == BlackboxSessionIdResponse.Status.NEED_RESET;
        boolean secure = resp.getAuth().map(BlackboxSessionIdResponse.Auth::isSecure).orElse(false);
        ValidSessionId valid = new ValidSessionId(resp.getUid().get().getValue(), needReset, secure);
        return Mono.just(CheckedSessionId.valid(valid));
    }

    private Mono<CheckedOAuthToken> fromOAuthResponse(BlackboxOAuthResponse resp) {
        if (resp.getException().isPresent()
                && resp.getException().get().getId() != BlackboxSessionIdResponse.BlackboxException.OK) {
            return Mono.error(new BlackboxException(resp.getException().get().getId(),
                    resp.getException().get().getValue(), resp.getError().orElse("")));
        }
        if (resp.getStatus().isEmpty()) {
            return Mono.error(new IllegalStateException("Missing status in blackbox response"));
        }
        if (resp.getStatus().get().getId() != BlackboxSessionIdResponse.Status.VALID) {
            String error = resp.getError().orElse("");
            InvalidOAuthToken.Status status = new InvalidOAuthToken.Status(resp.getStatus().get().getId(),
                    resp.getStatus().get().getValue());
            InvalidOAuthToken invalid = new InvalidOAuthToken(error, status);
            return Mono.just(CheckedOAuthToken.invalid(invalid));
        }
        if (resp.getUid().isEmpty()) {
            return Mono.error(new IllegalStateException("Missing uid in blackbox response"));
        }
        if (resp.getOauth().isEmpty()) {
            return Mono.error(new IllegalStateException("Missing oauth section in blackbox response"));
        }
        Set<String> scopes = Splitter.on(" ").splitToStream(resp.getOauth().get().getScope())
                .collect(Collectors.toSet());
        ValidOAuthToken valid = new ValidOAuthToken(resp.getUid().get().getValue(),
                resp.getOauth().get().getClientId(), resp.getOauth().get().getClientName(), scopes);
        return Mono.just(CheckedOAuthToken.valid(valid));
    }

    private static WebClient createWebClient(int connectTimeoutMillis,
                                             long readTimeoutMillis,
                                             long writeTimeoutMillis) {
        HttpClient httpClient = HttpClient.create()
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                .resolver(ResolverHolder.RESOLVER_INSTANCE)
                .doOnConnected(connection -> {
                    connection.addHandlerLast(new ReadTimeoutHandler(readTimeoutMillis, TimeUnit.MILLISECONDS));
                    connection.addHandlerLast(new WriteTimeoutHandler(writeTimeoutMillis, TimeUnit.MILLISECONDS));
                });
        return WebClient.builder()
                .clientConnector(new YaReactorClientHttpConnector(httpClient))
                .filter(ExchangeFilterFunctions.limitResponseSize(RESPONSE_SIZE_LIMIT))
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .maxInMemorySize(Math.toIntExact(RESPONSE_SIZE_LIMIT)))
                .build();
    }

    private Mono<WebClientResponseException> createException(ClientResponse clientResponse, String requestUri) {
        return DataBufferUtils.join(clientResponse.body(BodyExtractors.toDataBuffers())).map(dataBuffer -> {
            byte[] bytes = new byte[dataBuffer.readableByteCount()];
            dataBuffer.read(bytes);
            DataBufferUtils.release(dataBuffer);
            return bytes;
        })
        .defaultIfEmpty(EMPTY)
        .onErrorReturn(IllegalStateException.class::isInstance, EMPTY)
        .map(bodyBytes -> {
            Charset charset = clientResponse.headers().contentType()
                    .map(MimeType::getCharset)
                    .orElse(null);
            int statusCode = clientResponse.rawStatusCode();
            HttpStatus httpStatus = HttpStatus.resolve(statusCode);
            if (httpStatus != null) {
                final String message = initMessage(statusCode, httpStatus.getReasonPhrase(), requestUri);
                return new WebClientResponseException(message, statusCode, httpStatus.getReasonPhrase(),
                        clientResponse.headers().asHttpHeaders(), bodyBytes, charset, null);
            } else {
                return new UnknownHttpStatusCodeException(statusCode,
                        clientResponse.headers().asHttpHeaders(), bodyBytes, charset, null);
            }
        });
    }

    private String initMessage(int status, String reasonPhrase, String uri) {
        URI securedUri = UriComponentsBuilder.fromHttpUrl(uri)
                .replaceQueryParam("oauth_token")
                .replaceQueryParam("sessionid")
                .replaceQueryParam("sslsessionid").build(Map.of());
        return status + " " + reasonPhrase + " from GET " + securedUri;
    }

}
