package ru.yandex.intranet.d.web.security.impl;

import org.springframework.security.authentication.ReactiveAuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.web.security.model.YaAuthenticationToken;
import ru.yandex.intranet.d.web.security.model.YaCredentials;

/**
 * Reactive Yandex authentication manager. Loads and validates user details.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ReactiveYaAuthenticationManager implements ReactiveAuthenticationManager {

    private final ReactiveYaUserDetailsService userDetailsService;
    private final YaUserDetailsChecker userDetailsChecker;

    public ReactiveYaAuthenticationManager(
            ReactiveYaUserDetailsService userDetailsService,
            YaUserDetailsChecker userDetailsChecker) {
        this.userDetailsService = userDetailsService;
        this.userDetailsChecker = userDetailsChecker;
    }

    @Override
    public Mono<Authentication> authenticate(Authentication authentication) {
        return Mono.just(authentication)
                .filter(this::supports)
                .cast(YaAuthenticationToken.class)
                .flatMap(userDetailsService::findByAuthenticationToken)
                .switchIfEmpty(Mono.error(() -> new UsernameNotFoundException("User not found")))
                .doOnNext(userDetailsChecker::check)
                .map(details -> {
                    YaAuthenticationToken result = new YaAuthenticationToken(
                            details, (YaCredentials) authentication.getCredentials(), details.getAuthorities());
                    result.setDetails(authentication.getDetails());
                    return result;
                });
    }

    private boolean supports(Authentication authentication) {
        return YaAuthenticationToken.class.isAssignableFrom(authentication.getClass());
    }

}
