package ru.yandex.intranet.d.web.security.impl;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.users.UsersLoader;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.web.security.model.YaAuthenticationToken;
import ru.yandex.intranet.d.web.security.model.YaPrincipal;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Reactive Yandex user details service. Loads user details.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ReactiveYaUserDetailsService {

    private final UsersLoader usersLoader;
    private final ProvidersLoader providersLoader;
    private final Map<Long, String> tvmSourceIdRoleMap;

    public ReactiveYaUserDetailsService(
            UsersLoader usersLoader, ProvidersLoader providersLoader,
            @Value("${solomon.tvmSourceId}") long solomonTvmSourceId,
            @Value("${idm.tvmSourceId}") long idmTvmSourceId,
            @Value("${hardwareOrderService.tvmSourceId}") long hardwareOrderServiceTvmSrouceId,
            @Value("${abc.tvmSourceId}") long abcTvmSourceId) {

        this.usersLoader = usersLoader;
        this.providersLoader = providersLoader;

        tvmSourceIdRoleMap = ImmutableMap.of(
                solomonTvmSourceId, YaUserDetails.SOLOMON_FETCHER_ROLE,
                idmTvmSourceId, YaUserDetails.IDM_SERVICE_ROLE,
                hardwareOrderServiceTvmSrouceId, YaUserDetails.HARDWARE_ORDER_SERVICE_ROLE,
                abcTvmSourceId, YaUserDetails.ABC_SERVICE_ROLE
        );
    }

    public Mono<YaUserDetails> findByAuthenticationToken(YaAuthenticationToken authenticationToken) {
        YaPrincipal yaPrincipal = (YaPrincipal) authenticationToken.getPrincipal();
        if (yaPrincipal.getUid().isPresent()) {
            return getDetailsByUid(yaPrincipal.getUid().get(), yaPrincipal);
        } else if (yaPrincipal.getTvmServiceId().isPresent()) {
            return getDetailsByTvmServiceId(yaPrincipal.getTvmServiceId().get(), yaPrincipal);
        } else {
            return Mono.empty();
        }
    }

    private Mono<YaUserDetails> getDetailsByUid(String uid, YaPrincipal yaPrincipal) {
        return usersLoader.getUserByPassportUidImmediate(uid, Tenants.DEFAULT_TENANT_ID).flatMap(user -> {
            if (user.isPresent() && !user.get().isDeleted()) {
                Set<SimpleGrantedAuthority> authorities
                        = Set.of(new SimpleGrantedAuthority(YaUserDetails.USER_ROLE));
                YaUserDetails userDetails = new YaUserDetails(
                        uid,
                        yaPrincipal.getTvmServiceId().orElse(null),
                        yaPrincipal.getOAuthClientId().orElse(null),
                        yaPrincipal.getOAuthClientName().orElse(null),
                        yaPrincipal.getScopes(), user.get(), null, authorities);
                return Mono.just(userDetails);
            } else {
                return Mono.empty();
            }
        });
    }

    private Mono<YaUserDetails> getDetailsByTvmServiceId(long tvmServiceId, YaPrincipal yaPrincipal) {
        Mono<List<ProviderModel>> providerMono =
                providersLoader.getProviderBySourceTvmIdImmediate(tvmServiceId, Tenants.DEFAULT_TENANT_ID);

        boolean isSpecialSource = tvmSourceIdRoleMap.containsKey(tvmServiceId);
        if (isSpecialSource) {
            providerMono = providerMono.onErrorReturn(List.of());
        }

        return providerMono
                .flatMap(providers -> {
                    if (!providers.isEmpty() && !providers.stream().allMatch(ProviderModel::isDeleted)) {
                        Set<SimpleGrantedAuthority> authorities = new HashSet<>();
                        if (isSpecialSource) {
                            authorities.add(new SimpleGrantedAuthority(tvmSourceIdRoleMap.get(tvmServiceId)));
                        }
                        authorities.add(new SimpleGrantedAuthority(YaUserDetails.SERVICE_ROLE));
                        YaUserDetails userDetails = new YaUserDetails(null, tvmServiceId, null, null,
                                yaPrincipal.getScopes(), null, providers, authorities);
                        return Mono.just(userDetails);
                    } else if (isSpecialSource) {
                        return getYaUserDetailsWithRole(tvmServiceId, yaPrincipal,
                                tvmSourceIdRoleMap.get(tvmServiceId));
                    } else {
                        return Mono.empty();
                    }
                });
    }

    private Mono<YaUserDetails> getYaUserDetailsWithRole(long tvmServiceId, YaPrincipal yaPrincipal, String role) {
        Set<SimpleGrantedAuthority> authorities = Set
                .of(new SimpleGrantedAuthority(role));
        YaUserDetails userDetails = new YaUserDetails(null, tvmServiceId, null, null,
                yaPrincipal.getScopes(), null, null, authorities);
        return Mono.just(userDetails);
    }

}
