package ru.yandex.intranet.d.web.security.impl;

import java.net.InetSocketAddress;
import java.util.Optional;

import org.springframework.context.annotation.Profile;
import org.springframework.http.HttpCookie;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.server.authentication.ServerAuthenticationConverter;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.web.log.AccessLogAttributesProducer;
import ru.yandex.intranet.d.web.security.blackbox.BlackboxAuthChecker;
import ru.yandex.intranet.d.web.security.model.YaAuthenticationToken;
import ru.yandex.intranet.d.web.security.tvm.TvmTicketChecker;

/**
 * Extracts authentication headers, authenticate with TVM or blackbox.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Profile({"dev", "testing", "production"})
@Component("yaAuthenticationConverter")
public class YaAuthenticationConverter implements ServerAuthenticationConverter {

    private final AccessLogAttributesProducer accessLogAttributesProducer;
    private final TvmTicketChecker tvmTicketChecker;
    private final BlackboxAuthChecker blackboxAuthChecker;

    public YaAuthenticationConverter(AccessLogAttributesProducer accessLogAttributesProducer,
                                     TvmTicketChecker tvmTicketChecker, BlackboxAuthChecker blackboxAuthChecker) {
        this.accessLogAttributesProducer = accessLogAttributesProducer;
        this.tvmTicketChecker = tvmTicketChecker;
        this.blackboxAuthChecker = blackboxAuthChecker;
    }

    @Override
    public Mono<Authentication> convert(ServerWebExchange exchange) {
        String userTicket = exchange.getRequest().getHeaders().getFirst("X-Ya-User-Ticket");
        String serviceTicket = exchange.getRequest().getHeaders().getFirst("X-Ya-Service-Ticket");
        String oauthToken = Optional.ofNullable(exchange.getRequest().getHeaders().getFirst("Authorization"))
                .flatMap(this::getToken).orElse(null);
        String sessionId = Optional.ofNullable(exchange.getRequest().getCookies().getFirst("Session_id"))
                .map(HttpCookie::getValue).orElse(null);
        String sslSessionId = Optional.ofNullable(exchange.getRequest().getCookies().getFirst("sessionid2"))
                .map(HttpCookie::getValue).orElse(null);

        if (userTicket != null && serviceTicket != null) {
            return tvmTicketChecker.checkUser(userTicket, serviceTicket)
                    .doOnSuccess(auth -> rememberAuth(exchange, auth)).cast(Authentication.class);
        } else if (serviceTicket != null) {
            return tvmTicketChecker.checkService(serviceTicket)
                    .doOnSuccess(auth -> rememberAuth(exchange, auth)).cast(Authentication.class);
        } else if (oauthToken != null) {
            String userIp = getSourceIp(exchange).orElse("127.0.0.1");
            return blackboxAuthChecker.checkOauthToken(oauthToken, userIp)
                    .doOnSuccess(auth -> rememberAuth(exchange, auth)).cast(Authentication.class);
        } else if (sessionId != null && !sessionId.isEmpty() && sslSessionId != null && !sslSessionId.isEmpty()) {
            String userIp = getSourceIp(exchange).orElse("127.0.0.1");
            return blackboxAuthChecker.checkSessionId(sessionId, sslSessionId, userIp)
                    .doOnSuccess(auth -> rememberAuth(exchange, auth)).cast(Authentication.class);
        } else {
            return Mono.empty();
        }
    }

    private Optional<String> getToken(String value) {
        if (value.startsWith("OAuth ") && value.length() > 6) {
            return Optional.of(value.substring(6));
        }
        return Optional.empty();
    }

    private Optional<String> getSourceIp(ServerWebExchange exchange) {
        String forwardedForY = exchange.getRequest().getHeaders().getFirst("X-Forwarded-For-Y");
        if (forwardedForY != null && !forwardedForY.isBlank()) {
            return Optional.of(forwardedForY);
        }
        return Optional.ofNullable(exchange.getRequest().getRemoteAddress())
                .map(InetSocketAddress::getHostString).map(this::prepareIp);
    }

    private String prepareIp(String value) {
        if (value.contains("%")) {
            int lastSeparatorIndex = value.lastIndexOf("%");
            return value.substring(0, lastSeparatorIndex);
        }
        return value;
    }

    private void rememberAuth(ServerWebExchange exchange, YaAuthenticationToken authentication) {
        if (authentication != null) {
            accessLogAttributesProducer.onAuthenticationToken(exchange, authentication);
        }
    }

}
