package ru.yandex.intranet.d.web.security.impl;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.codec.HttpMessageWriter;
import org.springframework.http.codec.ServerCodecConfigurer;
import org.springframework.lang.NonNull;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.web.server.authorization.ServerAccessDeniedHandler;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.server.ServerResponse;
import org.springframework.web.reactive.result.view.ViewResolver;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.i18n.Locales;

/**
 * Server access denied handler.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class YaServerAccessDeniedHandler implements ServerAccessDeniedHandler {

    private final List<HttpMessageWriter<?>> messageWriters;
    private final List<ViewResolver> viewResolvers;
    private final MessageSource messages;

    public YaServerAccessDeniedHandler(ObjectProvider<ViewResolver> viewResolvers,
                                       ServerCodecConfigurer serverCodecConfigurer,
                                       @Qualifier("messageSource") MessageSource messages) {
        this.viewResolvers = viewResolvers.orderedStream().collect(Collectors.toList());
        this.messageWriters = serverCodecConfigurer.getWriters();
        this.messages = messages;
    }

    @Override
    public Mono<Void> handle(ServerWebExchange exchange, AccessDeniedException denied) {
        HttpStatus errorStatus = HttpStatus.FORBIDDEN;
        Locale locale = Optional.ofNullable(exchange.getLocaleContext().getLocale())
                .orElse(Locales.ENGLISH);
        Map<String, Object> errorAttributes = new LinkedHashMap<>();
        errorAttributes.put("status", errorStatus.value());
        errorAttributes.put("error", errorStatus.getReasonPhrase());
        errorAttributes.put("message",
                messages.getMessage("errors.access.denied", null, locale));
        return ServerResponse.status(errorStatus).contentType(MediaType.APPLICATION_JSON)
                .body(BodyInserters.fromValue(errorAttributes)).flatMap(response -> {
            exchange.getResponse().getHeaders().setContentType(response.headers().getContentType());
            return response.writeTo(exchange, new ResponseContext(messageWriters, viewResolvers));
        });
    }

    private static class ResponseContext implements ServerResponse.Context {

        private final List<HttpMessageWriter<?>> messageWriters;
        private final List<ViewResolver> viewResolvers;

        private ResponseContext(List<HttpMessageWriter<?>> messageWriters, List<ViewResolver> viewResolvers) {
            this.messageWriters = messageWriters;
            this.viewResolvers = viewResolvers;
        }

        @NonNull
        @Override
        public List<HttpMessageWriter<?>> messageWriters() {
            return messageWriters;
        }

        @NonNull
        @Override
        public List<ViewResolver> viewResolvers() {
            return viewResolvers;
        }

    }

}
