package ru.yandex.intranet.d.web.security.model;

import java.util.Collection;
import java.util.List;
import java.util.Objects;

import org.springframework.security.core.Authentication;
import org.springframework.security.core.CredentialsContainer;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;

/**
 * Yandex authentication info, contains uid and/or tvm id before user details are loaded and user details afterwards.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YaAuthenticationToken implements Authentication, CredentialsContainer {

    private final Object principal;
    private final Object credentials;
    private final List<GrantedAuthority> authorities;
    private Object details;
    private boolean authenticated = false;

    public YaAuthenticationToken(YaPrincipal principal, YaCredentials credentials) {
        this.principal = principal;
        this.credentials = credentials;
        this.authorities = AuthorityUtils.NO_AUTHORITIES;
    }

    public YaAuthenticationToken(YaUserDetails principal, YaCredentials credentials,
                                 Collection<? extends GrantedAuthority> authorities) {
        this.principal = principal;
        this.credentials = credentials;
        this.authenticated = true;
        if (authorities == null) {
            this.authorities = AuthorityUtils.NO_AUTHORITIES;
        } else {
            if (authorities.stream().anyMatch(Objects::isNull)) {
                throw new IllegalArgumentException("Authorities collection cannot contain any null elements");
            }
            this.authorities = List.copyOf(authorities);
        }
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        return authorities;
    }

    @Override
    public Object getCredentials() {
        return credentials;
    }

    @Override
    public Object getDetails() {
        return null;
    }

    @Override
    public Object getPrincipal() {
        return principal;
    }

    @Override
    public boolean isAuthenticated() {
        return authenticated;
    }

    @Override
    public void setAuthenticated(boolean isAuthenticated) throws IllegalArgumentException {
        if (isAuthenticated) {
            throw new IllegalArgumentException("Can not authenticate this token");
        }
        authenticated = false;
    }

    @Override
    public String getName() {
        if (getPrincipal() instanceof YaUserDetails) {
            return ((YaUserDetails) getPrincipal()).getUsername();
        }
        if (getPrincipal() instanceof YaPrincipal) {
            return ((YaPrincipal) getPrincipal()).getName();
        }
        return Objects.toString(getPrincipal(), "");
    }

    @Override
    public void eraseCredentials() {
        if (getCredentials() instanceof CredentialsContainer) {
            ((CredentialsContainer) getCredentials()).eraseCredentials();
        }
        if (getPrincipal() instanceof CredentialsContainer) {
            ((CredentialsContainer) getPrincipal()).eraseCredentials();
        }
        if (details instanceof CredentialsContainer) {
            ((CredentialsContainer) details).eraseCredentials();
        }
    }

    public void setDetails(Object details) {
        this.details = details;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YaAuthenticationToken that = (YaAuthenticationToken) o;
        return authenticated == that.authenticated &&
                Objects.equals(principal, that.principal) &&
                Objects.equals(credentials, that.credentials) &&
                Objects.equals(authorities, that.authorities) &&
                Objects.equals(details, that.details);
    }

    @Override
    public int hashCode() {
        return Objects.hash(principal, credentials, authorities, details, authenticated);
    }

    @Override
    public String toString() {
        return "YaAuthenticationToken{" +
                "principal=" + principal +
                ", credentials=" + credentials +
                ", authorities=" + authorities +
                ", details=" + details +
                ", authenticated=" + authenticated +
                '}';
    }

}
