package ru.yandex.intranet.d.web.security.model;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Yandex user details.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YaUserDetails implements UserDetails {

    public static final String USER_ROLE = "ROLE_USER";
    public static final String SERVICE_ROLE = "ROLE_SERVICE";
    public static final String SOLOMON_FETCHER_ROLE = "ROLE_SOLOMON_FETCHER";
    public static final String IDM_SERVICE_ROLE = "ROLE_IDM_SERVICE";
    public static final String HARDWARE_ORDER_SERVICE_ROLE = "ROLE_HARDWARE_ORDER_SERVICE";
    public static final String ABC_SERVICE_ROLE = "ROLE_ABC_SERVICE";

    private final String uid;
    private final Long tvmServiceId;
    private final String oauthClientId;
    private final String oauthClientName;
    private final Set<String> scopes;
    private final UserModel user;
    private final List<ProviderModel> providers;
    private final Set<? extends GrantedAuthority> authorities;

    @SuppressWarnings("ParameterNumber")
    public YaUserDetails(String uid, Long tvmServiceId, String oauthClientId, String oauthClientName,
                         Set<String> scopes, UserModel user, List<ProviderModel> providers,
                         Set<? extends GrantedAuthority> authorities) {
        this.uid = uid;
        this.tvmServiceId = tvmServiceId;
        this.oauthClientId = oauthClientId;
        this.oauthClientName = oauthClientName;
        this.scopes = scopes;
        this.user = user;
        this.providers = providers != null ? providers : List.of();
        this.authorities = authorities;
    }

    public static YaUserDetails fromUser(UserModel user) {
        return new YaUserDetails(user.getPassportUid().orElse(null), null, null, null, Set.of(), user, null,
                Set.of(new SimpleGrantedAuthority(YaUserDetails.USER_ROLE)));
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        return authorities;
    }

    @Override
    public String getPassword() {
        return null;
    }

    @Override
    public String getUsername() {
        if (user != null) {
            return user.getPassportLogin().orElse(user.getId());
        } else if (!providers.isEmpty()) {
            return providers.get(0).getId();
        }
        return "-";
    }

    @Override
    public boolean isAccountNonExpired() {
        return true;
    }

    @Override
    public boolean isAccountNonLocked() {
        return true;
    }

    @Override
    public boolean isCredentialsNonExpired() {
        return true;
    }

    @Override
    public boolean isEnabled() {
        if (user != null) {
            return !user.getStaffDismissed().orElse(false);
        }
        return true;
    }

    public Optional<String> getUid() {
        return Optional.ofNullable(uid);
    }

    public Optional<Long> getTvmServiceId() {
        return Optional.ofNullable(tvmServiceId);
    }

    public Optional<String> getOAuthClientId() {
        return Optional.ofNullable(oauthClientId);
    }

    public Optional<String> getOAuthClientName() {
        return Optional.ofNullable(oauthClientName);
    }

    public Set<String> getScopes() {
        return scopes;
    }

    public Optional<UserModel> getUser() {
        return Optional.ofNullable(user);
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public Optional<YaUserDetails> toProvider(String providerId, TenantId tenantId) {
        return providers.stream()
                .filter(provider -> provider.getId().equals(providerId) && provider.getTenantId().equals(tenantId))
                .findFirst()
                .map(provider -> new YaUserDetails(
                        uid, tvmServiceId, oauthClientId, oauthClientName,
                        scopes, user, List.of(provider),
                        authorities
                ));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YaUserDetails that = (YaUserDetails) o;
        return Objects.equals(uid, that.uid) &&
                Objects.equals(tvmServiceId, that.tvmServiceId) &&
                Objects.equals(oauthClientId, that.oauthClientId) &&
                Objects.equals(oauthClientName, that.oauthClientName) &&
                Objects.equals(scopes, that.scopes) &&
                Objects.equals(user, that.user) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(authorities, that.authorities);
    }

    @Override
    public int hashCode() {
        return Objects.hash(uid, tvmServiceId, oauthClientId, oauthClientName, scopes, user, providers, authorities);
    }

    @Override
    public String toString() {
        return "YaUserDetails{" +
                "uid='" + uid + '\'' +
                ", tvmServiceId=" + tvmServiceId +
                ", oauthClientId='" + oauthClientId + '\'' +
                ", oauthClientName='" + oauthClientName + '\'' +
                ", scopes=" + scopes +
                ", user=" + user +
                ", providers=" + providers +
                ", authorities=" + authorities +
                '}';
    }
}
