package ru.yandex.intranet.d.web.security.tvm;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;

import ru.yandex.intranet.d.util.ResolverHolder;
import ru.yandex.intranet.d.util.http.YaReactorClientHttpConnector;
import ru.yandex.intranet.d.web.security.tvm.model.CheckedServiceTicket;
import ru.yandex.intranet.d.web.security.tvm.model.CheckedUserTicket;
import ru.yandex.intranet.d.web.security.tvm.model.InvalidServiceTicket;
import ru.yandex.intranet.d.web.security.tvm.model.InvalidUserTicket;
import ru.yandex.intranet.d.web.security.tvm.model.TvmEnvironment;
import ru.yandex.intranet.d.web.security.tvm.model.TvmStatus;
import ru.yandex.intranet.d.web.security.tvm.model.TvmTicket;
import ru.yandex.intranet.d.web.security.tvm.model.ValidServiceTicket;
import ru.yandex.intranet.d.web.security.tvm.model.ValidUserTicket;

/**
 * TVM daemon client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
@Profile({"dev", "testing", "production", "load-testing"})
public class TvmClient {

    private static final long RESPONSE_SIZE_LIMIT = 10485760L;

    private final WebClient client;
    private final String baseUrl;
    private final String authToken;
    private final String userAgent;

    public TvmClient(@Value("${tvm.client.connectTimeoutMs}") int connectTimeoutMillis,
                     @Value("${tvm.client.readTimeoutMs}") long readTimeoutMillis,
                     @Value("${tvm.client.writeTimeoutMs}") long writeTimeoutMillis,
                     @Value("${http.client.userAgent}") String userAgent,
                     TvmClientParams tvmClientParams) {
        this.client = createWebClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis);
        this.baseUrl = tvmClientParams.getBaseUrl();
        this.authToken = tvmClientParams.getAuthToken();
        this.userAgent = userAgent;
    }

    public boolean isInitialized() {
        return baseUrl != null && authToken != null;
    }

    public Mono<TvmStatus> ping() {
        if (!isInitialized()) {
            return Mono.error(new IllegalStateException("TVM client is not initialized"));
        }
        return client.get()
                .uri(UriComponentsBuilder.fromHttpUrl(baseUrl).path("/tvm/ping").toUriString())
                .header("Authorization", authToken)
                .accept(MediaType.TEXT_PLAIN)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(String.class).map(m -> new TvmStatus(TvmStatus.Status.OK, m));
                    } else if (r.rawStatusCode() == HttpStatus.PARTIAL_CONTENT.value()) {
                        return r.bodyToMono(String.class).map(m -> new TvmStatus(TvmStatus.Status.WARN, m));
                    } else if (r.rawStatusCode() == HttpStatus.INTERNAL_SERVER_ERROR.value()) {
                        return r.bodyToMono(String.class).map(m -> new TvmStatus(TvmStatus.Status.ERROR, m));
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    public Mono<Map<String, TvmTicket>> tickets(List<String> destinations) {
        return tickets(null, destinations);
    }

    public Mono<Map<String, TvmTicket>> tickets(String source, List<String> destinations) {
        if (!isInitialized()) {
            return Mono.error(new IllegalStateException("TVM client is not initialized"));
        }
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl).path("/tvm/tickets");
        if (source != null) {
            builder.queryParam("src", source);
        }
        builder.queryParam("dsts", String.join(",", destinations));
        return client.get()
                .uri(builder.toUriString())
                .header("Authorization", authToken)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(new ParameterizedTypeReference<Map<String, TvmTicket>>() { });
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    public Mono<String> keys() {
        if (!isInitialized()) {
            return Mono.error(new IllegalStateException("TVM client is not initialized"));
        }
        return client.get()
                .uri(UriComponentsBuilder.fromHttpUrl(baseUrl).path("/tvm/keys").toUriString())
                .header("Authorization", authToken)
                .accept(MediaType.TEXT_PLAIN)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(String.class);
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    public Mono<CheckedServiceTicket> checkServiceTicket(String serviceTicket) {
        return checkServiceTicket(null, serviceTicket);
    }

    public Mono<CheckedServiceTicket> checkServiceTicket(String destination, String serviceTicket) {
        if (!isInitialized()) {
            return Mono.error(new IllegalStateException("TVM client is not initialized"));
        }
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl).path("/tvm/checksrv");
        if (destination != null) {
            builder.queryParam("dst", destination);
        }
        return client.get()
                .uri(builder.toUriString())
                .header("Authorization", authToken)
                .header("X-Ya-Service-Ticket", serviceTicket)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(ValidServiceTicket.class).map(CheckedServiceTicket::valid);
                    } else if (r.rawStatusCode() == HttpStatus.FORBIDDEN.value()) {
                        return r.bodyToMono(InvalidServiceTicket.class).map(CheckedServiceTicket::invalid);
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    public Mono<CheckedUserTicket> checkUserTicket(String userTicket) {
        return checkUserTicket(null, userTicket);
    }

    public Mono<CheckedUserTicket> checkUserTicket(TvmEnvironment environmentOverride, String userTicket) {
        if (!isInitialized()) {
            return Mono.error(new IllegalStateException("TVM client is not initialized"));
        }
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(baseUrl).path("/tvm/checkusr");
        if (environmentOverride != null) {
            builder.queryParam("override_env", environmentOverride.getValue());
        }
        return client.get()
                .uri(builder.toUriString())
                .header("Authorization", authToken)
                .header("X-Ya-User-Ticket", userTicket)
                .accept(MediaType.APPLICATION_JSON)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .exchangeToMono(r -> {
                    if (r.rawStatusCode() == HttpStatus.OK.value()) {
                        return r.bodyToMono(ValidUserTicket.class).map(CheckedUserTicket::valid);
                    } else if (r.rawStatusCode() == HttpStatus.FORBIDDEN.value()) {
                        return r.bodyToMono(InvalidUserTicket.class).map(CheckedUserTicket::invalid);
                    } else {
                        return r.createException().flatMap(Mono::error);
                    }
                });
    }

    private static WebClient createWebClient(int connectTimeoutMillis,
                                             long readTimeoutMillis,
                                             long writeTimeoutMillis) {
        HttpClient httpClient = HttpClient.create()
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                .resolver(ResolverHolder.RESOLVER_INSTANCE)
                .doOnConnected(connection -> {
                    connection.addHandlerLast(new ReadTimeoutHandler(readTimeoutMillis, TimeUnit.MILLISECONDS));
                    connection.addHandlerLast(new WriteTimeoutHandler(writeTimeoutMillis, TimeUnit.MILLISECONDS));
                });
        return WebClient.builder()
                .clientConnector(new YaReactorClientHttpConnector(httpClient))
                .filter(ExchangeFilterFunctions.limitResponseSize(RESPONSE_SIZE_LIMIT))
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .maxInMemorySize(Math.toIntExact(RESPONSE_SIZE_LIMIT)))
                .build();
    }

}
