package ru.yandex.intranet.d.web.util;

import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.BillingMeta;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.services.ServiceReadOnlyState;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.web.model.FeatureStateDto;
import ru.yandex.intranet.d.web.model.ProviderDto;
import ru.yandex.intranet.d.web.model.folders.FolderDto;
import ru.yandex.intranet.d.web.model.folders.front.ResourceTypeDto;
import ru.yandex.intranet.d.web.model.providers.ProviderRelatedResourcesSettingsDto;
import ru.yandex.intranet.d.web.model.providers.ProviderRelatedResourcesSettingsResponseDto;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.providers.RelatedResourceDto;
import ru.yandex.intranet.d.web.model.providers.RelatedResourceMappingDto;
import ru.yandex.intranet.d.web.model.providers.RelatedResourcesForResourceDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto;
import ru.yandex.intranet.d.web.model.services.ServiceMinimalDto;

import static ru.yandex.intranet.d.services.units.UnitsComparator.getBaseUnit;

/**
 * ModelDtoConverter.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 22-09-2021
 */
public class ModelDtoConverter {
    private ModelDtoConverter() {
    }

    public static AccountsSpaceDto toDto(ExpandedAccountsSpaces<AccountSpaceModel> em, Locale locale) {
        return toDto(em.getAccountsSpaces(), em.getSegmentations(), em.getSegments(), locale);
    }

    public static FolderDto toDto(FolderModel model) {
        return new FolderDto(
                model.getId(),
                model.getVersion(),
                model.getServiceId(),
                model.getFolderType(),
                model.getDisplayName(),
                model.getDescription().orElse(null),
                model.isDeleted(),
                model.getTags()
        );
    }

    public static ServiceMinimalDto toDto(ServiceMinimalModel model, Locale locale) {
        return new ServiceMinimalDto(
                model.getId(),
                Locales.select(model.getNameEn(), model.getName(), locale),
                model.getSlug(),
                model.getState(),
                model.getReadOnlyState() != null ? model.getReadOnlyState() : ServiceReadOnlyState.UNKNOWN,
                model.isExportable()
        );
    }

    public static AccountsSpaceDto toDto(
            AccountSpaceModel model,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            Locale locale
    ) {
        return new AccountsSpaceDto(
                model.getId(), // id
                model.getOuterKeyInProvider(), // key
                model.getProviderId(), // providerId
                model.getVersion(), // version
                Locales.select(model.getNameEn(), model.getNameRu(), locale), // name
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale), // description
                model.isReadOnly(), // readOnly
                toSegmentations(model, resourceSegmentations, resourceSegments, locale), // segment
                model.getUiSettings().map(ProviderUISettingsDto::new).orElse(null),
                model.isSyncEnabled()
        );
    }

    public static Set<InnerResourceSegmentationSegmentDto> toSegmentations(
            AccountSpaceModel model,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            Locale locale
    ) {
        return model.getSegments().stream().map(s -> {
            InnerResourceSegmentationDto segmentation = toResourceSegmentation(
                    resourceSegmentations.get(s.getSegmentationId()), locale);
            InnerResourceSegmentDto segment = toResourceSegment(
                    resourceSegments.get(s.getSegmentId()), locale);
            return new InnerResourceSegmentationSegmentDto(segmentation, segment);
        }).collect(Collectors.toSet());
    }

    public static InnerResourceSegmentationDto toResourceSegmentation(ResourceSegmentationModel model, Locale locale) {
        return new InnerResourceSegmentationDto(model.getId(),
                model.getKey(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale),
                model.getGroupingOrder(),
                model.getUiSettings().map(SegmentationUISettingsDto::new).orElse(null));
    }

    public static InnerResourceSegmentDto toResourceSegment(ResourceSegmentModel model, Locale locale) {
        return new InnerResourceSegmentDto(model.getId(),
                model.getKey(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale),
                model.getUncommon().orElse(false));
    }

    public static ResourceTypeDto resourceTypeDtoFromModel(
            ResourceTypeModel resourceType,
            Locale locale,
            Map<String, UnitsEnsembleModel> unitsEnsembleMap
    ) {
        UnitsEnsembleModel unitsEnsemble = unitsEnsembleMap.get(resourceType.getUnitsEnsembleId());
        UnitModel baseUnit = getBaseUnit(unitsEnsemble);

        String ensembleName = Locales.select(unitsEnsemble.getNameEn(), unitsEnsemble.getNameRu(), locale);
        String baseUnitName = baseUnit.getKey();
        String name = Locales.select(resourceType.getNameEn(), resourceType.getNameRu(), locale);
        String description = Locales.select(resourceType.getDescriptionEn(), resourceType.getDescriptionRu(), locale);

        return new ResourceTypeDto(
                resourceType.getId(),
                resourceType.getProviderId(),
                resourceType.getKey(),
                name,
                description,
                ensembleName,
                baseUnitName,
                resourceType.getSortingOrder()
        );
    }

    public static ProviderDto providerDtoFromModel(ProviderModel providerModel, Locale locale) {
        String name = Locales.select(providerModel.getNameEn(), providerModel.getNameRu(), locale);
        String description = Locales.select(providerModel.getDescriptionEn(), providerModel.getDescriptionRu(), locale);
        return new ProviderDto(
                providerModel.getId(),
                name,
                description,
                providerModel.isDeleted(),
                providerModel.isReadOnly(),
                providerModel.isManaged(),
                providerModel.getKey(),
                providerModel.getBillingMeta().flatMap(BillingMeta::getMeteringKey).orElse(null),
                providerModel.isMultipleAccountsPerFolder(),
                providerModel.isAccountTransferWithQuota(),
                providerModel.isImportAllowed(),
                providerModel.isAccountsSpacesSupported(),
                providerModel.isSyncEnabled(),
                toDto(providerModel.getAccountsSettings()),
                providerModel.getRelatedResourcesByResourceId()
                        .map(ModelDtoConverter::toRelatedResourceMappingDto)
                        .orElse(null),
                providerModel.getReserveFolderId().orElse(null),
                providerModel.hasDefaultQuotas(),
                FeatureStateDto.fromModel(providerModel.isAllocatedSupported().orElse(null)),
                providerModel.getUiSettings().map(ProviderUISettingsDto::new).orElse(null)
        );
    }

    public static ru.yandex.intranet.d.web.model.providers.ProviderDto toProvider(ProviderModel model, Locale locale) {
        return new ru.yandex.intranet.d.web.model.providers.ProviderDto(model.getId(), model.getVersion(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale),
                model.getServiceId(), model.isReadOnly(), model.isManaged(), model.getKey(),
                model.getUiSettings().map(ProviderUISettingsDto::new).orElse(null));
    }

    private static ProviderDto.AccountsSettingsDto toDto(AccountsSettingsModel model) {
        return new ProviderDto.AccountsSettingsDto(
                model.isDisplayNameSupported(),
                model.isKeySupported(),
                model.isDeleteSupported(),
                model.isSoftDeleteSupported(),
                model.isMoveSupported(),
                model.isRenameSupported(),
                model.isPerAccountVersionSupported(),
                model.isPerProvisionVersionSupported(),
                model.isPerAccountLastUpdateSupported(),
                model.isPerProvisionLastUpdateSupported(),
                model.isOperationIdDeduplicationSupported(),
                model.isSyncCoolDownDisabled(),
                model.isRetryCoolDownDisabled(),
                model.getAccountsSyncPageSize(),
                model.isMoveProvisionSupported(),
                model.getMultipleReservesAllowed().orElse(false));
    }

    public static Map<String, RelatedResourceMappingDto> toRelatedResourceMappingDto(
            Map<String, RelatedResourceMapping> relatedResourcesByResourceId) {
        return relatedResourcesByResourceId.entrySet().stream()
                .map(e -> Tuples.of(e.getKey(), RelatedResourceMappingDto.fromRelatedResourceMapping(e.getValue())))
                .collect(Collectors.toMap(Tuple2::getT1, Tuple2::getT2));
    }

    public static ProviderRelatedResourcesSettingsResponseDto toRelatedResources(ProviderModel provider) {
        Set<RelatedResourcesForResourceDto> relatedResourcesByResource = provider.getRelatedResourcesByResourceId()
                .orElse(Map.of()).entrySet().stream().map(e -> new RelatedResourcesForResourceDto(e.getKey(),
                        e.getValue().getRelatedCoefficientMap().entrySet().stream()
                                .map(ie -> new RelatedResourceDto(ie.getKey(), ie.getValue().getNumerator(),
                                        ie.getValue().getDenominator())).collect(Collectors.toSet())))
                .collect(Collectors.toSet());
        return new ProviderRelatedResourcesSettingsResponseDto(
                new ProviderRelatedResourcesSettingsDto(relatedResourcesByResource), provider.getVersion());
    }
}
