import copy

from django_pgaas import HostManager
from django_replicated.settings import *  # noqa F403

from .base import *  # noqa
from .base import env

# GENERAL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#secret-key
SECRET_KEY = env("DJANGO_SECRET_KEY", default="photoalbum")
# https://docs.djangoproject.com/en/dev/ref/settings/#allowed-hosts
ALLOWED_HOSTS = env.list("DJANGO_ALLOWED_HOSTS", default=["*"])

# DATABASES
# ------------------------------------------------------------------------------
POSTGRES_HOST_JSON = env.json("POSTGRES_HOST_JSON", default={})
DEPLOY_NODE_DC = env("DEPLOY_NODE_DC", default="myt")
if POSTGRES_HOST_JSON:
    host_manager = HostManager(
        list(POSTGRES_HOST_JSON.items()),
        current_dc=DEPLOY_NODE_DC,
    )
    DATABASES["default"]["HOST"] = host_manager.host_string
else:
    DATABASES["default"]["HOST"] = env("POSTGRES_HOST")

DATABASES["default"]["PORT"] = env.int("POSTGRES_PORT", default="6432")
DATABASES["default"]["OPTIONS"] = {
    "sslmode": "verify-full",
    "sslrootcert": "/app/.postgres/root.crt",
    "target_session_attrs": "read-write",
}
DATABASES["default"]["DISABLE_SERVER_SIDE_CURSORS"] = True

DATABASES["slave"] = copy.deepcopy(DATABASES["default"])
DATABASES["slave"]["OPTIONS"]["target_session_attrs"] = "any"


# REPLICATED
# ------------------------------------------------------------------------------
REPLICATED_DATABASE_SLAVES = ["slave"]
DATABASE_ROUTERS = ["django_replicated.router.ReplicationRouter"]
REPLICATED_VIEWS_OVERRIDES = {
    "/admin/*": "master",
}

# ZDM
# ------------------------------------------------------------------------------
PGAAS_USE_ZDM = True

# REDIS
# ------------------------------------------------------------------------------
REDIS_SENTINELS = env.json("REDIS_SENTINELS", default=[])
REDIS_SERVICE_NAME = env("REDIS_SERVICE_NAME", default="photoalbum")
REDIS_PASSWORD = env("REDIS_PASSWORD", default=None)
REDIS_MAX_CONNECTIONS = env("REDIS_MAX_CONNECTIONS", default=100)

DJANGO_REDIS_CONNECTION_FACTORY = "django_redis.pool.SentinelConnectionFactory"

# CACHES
# ------------------------------------------------------------------------------
CACHES = {
    "default": {
        "BACKEND": "django_redis.cache.RedisCache",
        "LOCATION": "redis://{service_name}/{db}".format(
            service_name=REDIS_SERVICE_NAME,
            db=0,
        ),
        "OPTIONS": {
            "CLIENT_CLASS": "django_redis.client.SentinelClient",
            "CONNECTION_POOL_KWARGS": {
                "max_connections": REDIS_MAX_CONNECTIONS,
                "retry_on_timeout": True,
            },
            "SENTINELS": REDIS_SENTINELS,
            "PASSWORD": REDIS_PASSWORD,
            # Mimicing memcache behavior.
            # https://github.com/jazzband/django-redis#memcached-exceptions-behavior
            "IGNORE_EXCEPTIONS": True,
        },
    }
}

# SECURITY
# ------------------------------------------------------------------------------
# # https://docs.djangoproject.com/en/dev/ref/settings/#secure-proxy-ssl-header
# SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")
# # https://docs.djangoproject.com/en/dev/ref/settings/#secure-ssl-redirect
# SECURE_SSL_REDIRECT = env.bool("DJANGO_SECURE_SSL_REDIRECT", default=True)
# https://docs.djangoproject.com/en/dev/ref/settings/#session-cookie-secure
SESSION_COOKIE_SECURE = True
# https://docs.djangoproject.com/en/dev/ref/settings/#csrf-cookie-secure
CSRF_COOKIE_SECURE = True
# # https://docs.djangoproject.com/en/dev/topics/security/#ssl-https
# # https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-seconds
# SECURE_HSTS_SECONDS = 60
# # https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-include-subdomains
# SECURE_HSTS_INCLUDE_SUBDOMAINS = env.bool(
#     "DJANGO_SECURE_HSTS_INCLUDE_SUBDOMAINS", default=True
# )
# # https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-preload
# SECURE_HSTS_PRELOAD = env.bool("DJANGO_SECURE_HSTS_PRELOAD", default=True)
# # https://docs.djangoproject.com/en/dev/ref/middleware/#x-content-type-options-nosniff
# SECURE_CONTENT_TYPE_NOSNIFF = env.bool(
#     "DJANGO_SECURE_CONTENT_TYPE_NOSNIFF", default=True
# )

# STATIC
# ------------------------
STATICFILES_STORAGE = "whitenoise.storage.CompressedManifestStaticFilesStorage"

# MEDIA
# ------------------------------------------------------------------------------
DEFAULT_FILE_STORAGE = "storages.backends.s3boto3.S3Boto3Storage"

# TEMPLATES
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#templates
TEMPLATES[-1]["OPTIONS"]["loaders"] = [  # type: ignore[index] # noqa F405
    (
        "django.template.loaders.cached.Loader",
        [
            "django.template.loaders.filesystem.Loader",
            "django.template.loaders.app_directories.Loader",
        ],
    )
]

# EMAIL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#default-from-email
DEFAULT_FROM_EMAIL = env(
    "DJANGO_DEFAULT_FROM_EMAIL",
    default="photoalbum <noreply@photoalbum.yandex-team.ru>",
)
# https://docs.djangoproject.com/en/dev/ref/settings/#server-email
SERVER_EMAIL = env("DJANGO_SERVER_EMAIL", default=DEFAULT_FROM_EMAIL)
# https://docs.djangoproject.com/en/dev/ref/settings/#email-subject-prefix
EMAIL_SUBJECT_PREFIX = env(
    "DJANGO_EMAIL_SUBJECT_PREFIX",
    default="[photoalbum]",
)

# LOGGING
# ------------------------------------------------------------------------------
LOGGING["formatters"]["json"] = {
    "()": "photoalbum.contrib.logger.format.JsonFormatter",
}


# Celery
# ------------------------------------------------------------------------------
if REDIS_SENTINELS:
    CELERY_BROKER_URL = ";".join(
        [
            f"sentinel://:{REDIS_PASSWORD}@{host}:{port}/1"
            for host, port in REDIS_SENTINELS
        ]
    )
    CELERY_BROKER_TRANSPORT_OPTIONS = {
        "master_name": REDIS_SERVICE_NAME,
    }
else:
    CELERY_BROKER_URL = env("CELERY_BROKER_URL")

# S3
AWS_ACCESS_KEY_ID = env("AWS_ACCESS_KEY_ID")
AWS_SECRET_ACCESS_KEY = env("AWS_SECRET_ACCESS_KEY")
AWS_S3_ENDPOINT_URL = env("AWS_S3_ENDPOINT_URL", default="https://s3.mds.yandex.net")

AWS_STORAGE_BUCKET_NAME = env("AWS_STORAGE_BUCKET_NAME")
AWS_QUERYSTRING_AUTH = False

# DEFACE
DEFACE_URL = env("DEFACE_URL", default="https://milab.yandex-team.ru/deface/detect")

# Your stuff...
# ------------------------------------------------------------------------------
DEPLOY_STAGE_ID = env("DEPLOY_STAGE_ID", default="photoalbum")
