import logging
from math import ceil
from typing import Optional

import requests
from django.core.exceptions import ImproperlyConfigured
from django.utils.functional import SimpleLazyObject

from photoalbum.utils import get_date_from_str

from .settings import (
    YADISK_API_TOKEN,
    YADISK_BASE_URL,
    YADISK_DEFAULT_TIMEOUT,
    YADISK_INIT_FOLDER,
)

logger = logging.getLogger(__name__)


class YaDiskClient:
    help = "Iterates over all files from YaDisk"

    def __init__(
        self,
        token,
        init_folder: str = "disk:/",
        base_url: str = None,
        timeout: Optional[int] = None,
        session=None,
    ):
        self.token = token
        self.init_folder = init_folder
        self.base_url = base_url or YADISK_BASE_URL
        self.timeout = timeout or YADISK_DEFAULT_TIMEOUT
        self.session = session or requests.Session()

        self.headers = {
            "Accept": "application/json",
            "Authorization": f"OAuth {self.token}",
        }

    @staticmethod
    def get_date_of_shot(item, default=None):
        date = item.get(
            "photoslice_time", item.get("exif", {}).get("date_time", default)
        )
        return get_date_from_str(date)

    def get_resources(self, path: str, **kwargs):
        url = f"{self.base_url}/disk/resources"
        params = {
            "path": path,
            **kwargs,
        }
        try:
            response = self.session.get(
                url=url, params=params, headers=self.headers, timeout=self.timeout
            )
            response.raise_for_status()
        except requests.exceptions.HTTPError as exc:
            if exc.response.status_code == 500:
                logger.error(exc)
            return {}
        except Exception as exc:
            logger.error(exc)
            return {}

        return response.json()

    def get_all_resources(self, path: str, **kwargs):
        limit = kwargs["limit"]
        offset = kwargs["offset"]

        response = self.get_resources(path, **kwargs)
        yield response

        total = response.get("_embedded", dict()).get("total", 0)
        if total > limit + offset:
            pages = ceil((total - offset) / limit)
            for page in range(1, pages):
                kwargs["offset"] = page * limit
                yield self.get_resources(path, **kwargs)

    def get_file(self, url: str, **kwargs):
        timeout = kwargs.pop("timeout", self.timeout)
        response = self.session.get(
            url=url, timeout=timeout, headers=self.headers, **kwargs
        )
        response.raise_for_status()
        return response

    def upload_file(self, media, **kwargs):
        url = f"{self.base_url}/disk/resources/upload"
        response = self.session.get(
            url=url,
            headers=self.headers,
            params={"path": media.path},
            **kwargs,
        )
        response.raise_for_status()
        upload_link = response.json()["href"]

        load_response = self.session.put(
            url=upload_link, headers=self.headers, files={"file": media.file}
        )
        load_response.raise_for_status()

    def create_folder(self, folder, **kwargs):
        url = f"{self.base_url}/disk/resources"
        response = self.session.put(
            url=url,
            headers=self.headers,
            params={"path": folder.path},
            **kwargs,
        )
        response.raise_for_status()


def get_client() -> YaDiskClient:
    if not YADISK_API_TOKEN:
        raise ImproperlyConfigured("Set the YADISK_API_TOKEN environment variable")

    return YaDiskClient(token=YADISK_API_TOKEN, init_folder=YADISK_INIT_FOLDER)


yadisk = SimpleLazyObject(lambda: get_client())
