import humanize
import requests
from django.contrib import admin
from django.urls import reverse
from django.utils.html import format_html, mark_safe
from django.utils.translation import gettext_lazy as _
from django_object_actions import DjangoObjectActions
from django_reverse_admin import ReverseModelAdmin
from mptt.admin import MPTTModelAdmin

from config.settings.base import YADISK_INIT_FOLDER
from photoalbum.contrib.yadisk import yadisk
from photoalbum.files.tasks import get_user_on_photo

from .constants import MB
from .models import (
    Author,
    Copyright,
    Folder,
    MediaFile,
    Photo,
    PhotoDuplicate,
    Tag,
    UserOnPhoto,
    Video,
)


class EXIFExistsFilter(admin.SimpleListFilter):
    title = _("EXIF")
    parameter_name = "EXIF"

    def lookups(self, request, model_admin):
        return (
            ("exif", _("есть EXIF")),
            ("no_exif", _("нет EXIF")),
        )

    def queryset(self, request, queryset):
        if self.value() == "exif":
            return queryset.filter(exif__has_key="date_time")
        elif self.value() == "no_exif":
            return queryset.exclude(exif__has_key="date_time")
        else:
            return queryset


class DownloadStatusFilter(admin.SimpleListFilter):
    title = _("загружено")
    parameter_name = "downloaded"

    def lookups(self, request, model_admin):
        return (
            ("downloaded", _("загружено")),
            ("not_downloaded", _("не загружено")),
        )

    def queryset(self, request, queryset):
        if self.value() == "downloaded":
            return queryset.exclude(file="")
        elif self.value() == "not_downloaded":
            return queryset.filter(file="")
        else:
            return queryset


class SizeFilter(admin.SimpleListFilter):
    title = _("размер больше")
    parameter_name = "size_is_bigger_than"

    def lookups(self, request, model_admin):
        return (
            (0, _("1 Гб")),
            (1, _("2 Гб")),
            (2, _("4 Гб")),
            (3, _("8 Гб")),
            (4, _("16 Гб")),
            (5, _("32 Гб")),
        )

    def queryset(self, request, queryset):
        if not self.value():
            return queryset
        else:
            return queryset.filter(size__gt=2 ** int(self.value()) * 1000 * MB)


class FolderFilter(admin.SimpleListFilter):
    title = _("папка")
    parameter_name = "folder"

    def lookups(self, request, model_admin):
        return [
            (obj.id, obj.path.replace(YADISK_INIT_FOLDER, ""))
            for obj in Folder.objects.all()
        ]

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(
                folder__in=Folder.objects.get(pk=self.value())
                .get_descendants(include_self=True)
                .values_list("pk", flat=True)
            )
        else:
            return queryset


class DetectedStatusFilter(admin.SimpleListFilter):
    title = _("распознано")
    parameter_name = "detected"

    def lookups(self, request, model_admin):
        return (
            ("detected", _("распознано")),
            ("not_detected", _("не распознано")),
        )

    def queryset(self, request, queryset):
        if self.value() == "detected":
            return queryset.filter(detected=True)
        elif self.value() == "not_detected":
            return queryset.filter(detected=False)
        else:
            return queryset


@admin.register(Author)
class AuthorAdmin(admin.ModelAdmin):
    fields = (
        "user",
        "first_name",
        "last_name",
        "email",
        "phone_number",
        "created",
        "modified",
    )
    list_display = (
        "user",
        "first_name",
        "last_name",
        "created",
        "modified",
    )
    readonly_fields = (
        "created",
        "modified",
    )
    list_select_related = ("user",)
    search_fields = ("user__username", "first_name", "last_name")
    raw_id_fields = ("user",)


@admin.register(Copyright)
class CopyrightAdmin(admin.ModelAdmin):
    fields = (
        "name",
        "slug",
        "is_active",
        "created",
        "modified",
    )
    list_display = (
        "name",
        "slug",
        "is_active",
        "created",
        "modified",
    )
    readonly_fields = (
        "created",
        "modified",
    )


@admin.register(Folder)
class FolderAdmin(MPTTModelAdmin):
    fields = (
        "name",
        "parent",
        "path",
        "mediafiles_link",
        "resource_id",
    )
    list_display = (
        "name",
        "path",
        "created",
        "modified",
    )
    readonly_fields = (
        "name",
        "parent",
        "path",
        "mediafiles_link",
        "resource_id",
    )
    mptt_level_indent = 20

    def mediafiles_link(self, obj):
        return format_html(
            '<a href="{}?folder={}" target="_blank">{}</a>',
            reverse("admin:files_mediafile_changelist"),
            obj.pk,
            "Показать все файлы в папке",
        )

    mediafiles_link.short_description = _("медиа")


@admin.register(MediaFile)
class MediaFileAdmin(DjangoObjectActions, admin.ModelAdmin):
    fields = (
        "filename",
        "natural_size",
        "path",
        "file",
        "tags",
        "copyright",
        "rights_comment",
        "author",
        "uploaded_by",
        "resource_id",
        "chksum",
        "media_type",
        "mime_type",
        "extra_info",
        "created",
        "modified",
    )
    list_display = (
        "filename",
        "natural_size",
        "path",
        "created",
        "modified",
        "downloaded",
    )

    def downloaded(self, obj):
        return bool(obj.file)

    downloaded.boolean = True
    downloaded.short_description = _("загружено")

    def natural_size(self, obj: MediaFile):
        return humanize.naturalsize(obj.size)

    natural_size.short_description = _("размер")
    natural_size.admin_order_field = "size"

    def update_link(self, request, obj):
        response = yadisk.get_resources(path=obj.path, fields="file")
        r = requests.get(response["file"], allow_redirects=True)
        open(obj.filename, "wb").write(r.content)

    update_link.label = "скачать файл"

    change_actions = ("update_link",)
    search_fields = ("path",)
    list_filter = (
        DownloadStatusFilter,
        "media_type",
        SizeFilter,
        FolderFilter,
        "copyright",
    )
    exclude = ("size",)
    filter_horizontal = ("tags",)
    raw_id_fields = ("copyright", "author", "uploaded_by")
    readonly_fields = (
        "resource_id",
        "path",
        "natural_size",
        "chksum",
        "media_type",
        "mime_type",
        "extra_info",
        "created",
        "modified",
    )


class UserOnPhotoInline(admin.TabularInline):
    model = UserOnPhoto
    fields = ("login", "match_score", "photo_preview", "verified")
    readonly_fields = (
        "match_score",
        "photo_preview",
    )
    extra = 0
    verbose_name_plural = _("пользователи на фото")

    def photo_preview(self, obj):
        return mark_safe('<img src="{}" height="200" />'.format(obj.staff_image_url))

    photo_preview.short_description = _("фото на стаффе")


@admin.register(Photo)
class PhotoAdmin(MediaFileAdmin, ReverseModelAdmin):
    fields = MediaFileAdmin.fields + (
        "show_preview",
        "photo_date",
        "exif",
        "detected",
    )

    list_display = MediaFileAdmin.list_display + (
        "photo_date",
        "exif_exists",
        "detected",
    )
    list_filter = (
        EXIFExistsFilter,
        DetectedStatusFilter,
        "location__country",
        "location__city",
    ) + MediaFileAdmin.list_filter
    date_hierarchy = "photo_date"
    inlines = [UserOnPhotoInline]
    inline_type = "stacked"
    inline_reverse = ["location"]

    def show_preview(self, obj: Photo):
        if obj.preview:
            return mark_safe('<img src="{}" />'.format(obj.preview.url))
        return ""

    show_preview.short_description = _("превью")

    def exif_exists(self, obj):
        return bool(obj.exif.get("date_time"))

    exif_exists.boolean = True
    exif_exists.short_description = _("EXIF")

    def detect_users(self, request, obj):
        get_user_on_photo.delay(obj.id)
        self.message_user(request, "Запущено распознавание людей на этом фото")

    detect_users.label = "распознать людей"

    change_actions = MediaFileAdmin.change_actions + ("detect_users",)
    exclude = MediaFileAdmin.exclude + ("preview",)
    readonly_fields = MediaFileAdmin.readonly_fields + (
        "exif",
        "show_preview",
    )


@admin.register(Video)
class VideoAdmin(MediaFileAdmin):
    fields = MediaFileAdmin.fields + (
        "video_date",
        "exif",
    )
    list_display = MediaFileAdmin.list_display + ("video_date",)
    readonly_fields = MediaFileAdmin.readonly_fields + ("exif",)
    date_hierarchy = "video_date"


@admin.register(UserOnPhoto)
class UserOnPhotoAdmin(admin.ModelAdmin):
    list_display = ("photo", "login", "created", "modified")
    raw_id_fields = ("photo",)
    search_fields = ("login",)


@admin.register(Tag)
class TagAdmin(admin.ModelAdmin):
    list_display = ("name",)


class PhotoDuplicateInline(admin.TabularInline):
    model = PhotoDuplicate.duplicates.through
    fields = ("photo_filename", "photo_path", "photo_preview")
    readonly_fields = fields
    extra = 0
    verbose_name_plural = _("дубликаты")

    def photo_filename(self, obj):
        return obj.photo.filename

    photo_filename.short_description = _("имя файла")

    def photo_path(self, obj):
        return obj.photo.path

    photo_path.short_description = _("путь")

    def photo_preview(self, obj):
        if obj.photo.preview:
            return mark_safe('<img src="{}" />'.format(obj.photo.preview.url))
        return ""

    photo_preview.short_description = _("превью")


@admin.register(PhotoDuplicate)
class PhotoDuplicateAdmin(admin.ModelAdmin):
    list_display = ("original", "duplicate_count", "created", "modified")
    raw_id_fields = ("original", "duplicates")
    filter_horizontal = ("duplicates",)

    fields = (
        "original",
        "photo_path",
        "photo_preview",
    )
    readonly_fields = fields

    inlines = [PhotoDuplicateInline]

    def photo_path(self, obj):
        return obj.original.path

    photo_path.short_description = _("путь")

    def photo_preview(self, obj):
        if obj.original.preview:
            return mark_safe('<img src="{}" />'.format(obj.original.preview.url))
        return ""

    photo_preview.short_description = _("превью")

    def duplicate_count(self, obj):
        return obj.duplicates.count()

    duplicate_count.short_description = _("количество дубликатов")
