import os
import re
import string
from datetime import datetime
from uuid import uuid4

import pymorphy2
from django.contrib.auth import get_user_model
from django.db import models, transaction
from django.utils.translation import gettext_lazy as _
from model_utils.models import TimeStampedModel
from mptt.fields import TreeForeignKey
from mptt.models import MPTTModel

from photoalbum.contrib.yadisk.settings import YADISK_INIT_FOLDER
from photoalbum.files.constants import FUNCTORS, REG_DATE

User = get_user_model()


def get_upload_path(obj, filename, prefix=None):
    if not prefix:
        prefix = obj.media_type if obj.media_type else "media"
    date_created = obj.created.strftime("%Y/%m/%d")
    real_name, ext = os.path.splitext(filename)
    filename = f"{prefix}/{date_created}/{real_name}_{uuid4().hex}{ext}"
    return filename


def get_preview_upload_path(obj, filename):
    return get_upload_path(obj, filename, "previews")


def filter_exclusions(tag):
    if any(
        [
            tag is None,
            len(tag) < 2,
            tag in string.punctuation,
        ]
    ):
        return False
    return True


def pos(word, morph=pymorphy2.MorphAnalyzer()):
    return morph.parse(word)[0].tag.POS


def create_tags(path, cached_tags):
    current_tags = path.replace(YADISK_INIT_FOLDER, "", 1).rsplit(os.sep, 1)[:-1]
    tag_ids = []
    with transaction.atomic():
        for tag in current_tags:
            matches = re.findall(REG_DATE, tag)
            date_tags = []
            text = tag
            if matches:
                for match in matches:
                    text = text.replace(match, "")
                    match = re.sub(r"[-./_,:]+", "/", match)
                    for date_format in ["%d/%m/%Y", "%d/%m/%y", "%Y/%m/%d", "%y/%m/%d"]:
                        try:
                            date = datetime.strptime(match, date_format).strftime(
                                "date:%d/%m/%Y"
                            )
                            date_tags.append(date)
                            break
                        except ValueError:
                            continue

            text = re.sub(rf"[{string.punctuation}]", " ", text).lower()
            word_tags = [
                word
                for word in text.split()
                if pos(word) not in FUNCTORS and filter_exclusions(word)
            ]

            for new_tag in date_tags + word_tags:
                if new_tag not in cached_tags:
                    tag_object = Tag.objects.create(name=new_tag)
                    cached_tags[new_tag] = tag_object.pk
                tag_ids.append(cached_tags[new_tag])
    return tag_ids


class Tag(TimeStampedModel):
    name = models.CharField(_("тег"), max_length=255, unique=True)

    class Meta:
        verbose_name = _("тег")
        verbose_name_plural = _("теги")

    def __str__(self):
        return self.name

    def __eq__(self, other):
        return self.name == other.name

    def __hash__(self):
        return hash(self.name)


class Location(TimeStampedModel):
    geo_coords = models.CharField(_("гео-координаты"), max_length=255, blank=True)
    country = models.CharField(_("страна"), max_length=255, blank=True)
    city = models.CharField(_("город"), max_length=255, blank=True)
    address = models.CharField(_("адрес"), max_length=255, blank=True)


class Author(TimeStampedModel):
    user = models.OneToOneField(
        User,
        verbose_name=_("пользователь"),
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    first_name = models.CharField(_("имя"), max_length=255, blank=True)
    last_name = models.CharField(_("фамилия"), max_length=255, blank=True)
    email = models.CharField(_("почта"), max_length=255, blank=True)
    phone_number = models.CharField(_("телефон"), max_length=255, blank=True)

    class Meta:
        verbose_name = _("автор")
        verbose_name_plural = _("авторы")

    def save(self, *args, **kwargs):
        if self.user:
            self.first_name = self.user.first_name
            self.last_name = self.user.last_name
            self.email = self.user.email
        return super().save(*args, **kwargs)


class Copyright(TimeStampedModel):
    name = models.CharField(_("права"), max_length=255)
    slug = models.SlugField(_("код"), max_length=255, unique=True)
    is_active = models.BooleanField(_("активно"), default=True)

    class Meta:
        verbose_name = _("права")
        verbose_name_plural = _("права")


class Folder(TimeStampedModel, MPTTModel):
    resource_id = models.CharField(
        _("resource id"),
        max_length=255,
        help_text=_("Поле resource_id из Я.Диска"),
        unique=True,
        blank=True,
    )
    name = models.CharField(_("имя папки"), max_length=255)
    path = models.CharField(_("путь"), max_length=1024)
    parent = TreeForeignKey(
        "self",
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name="children",
        db_index=True,
    )

    class Meta:
        verbose_name = _("папка")
        verbose_name_plural = _("папки")

    def __str__(self):
        return self.name


class MediaFile(TimeStampedModel):
    resource_id = models.CharField(
        _("resource id"),
        max_length=255,
        help_text=_("Поле resource_id из Я.Диска"),
        blank=True,
    )
    filename = models.CharField(_("имя файла"), max_length=255)
    path = models.CharField(_("путь"), max_length=1024, blank=True)
    size = models.PositiveBigIntegerField(_("размер"), default=0)
    file = models.FileField(
        _("файл"), upload_to=get_upload_path, blank=True, null=True, max_length=255
    )
    chksum = models.CharField(_("хеш сумма"), max_length=255, blank=True)
    tags = models.ManyToManyField(Tag, verbose_name=_("теги"), blank=True)
    media_type = models.CharField(_("тип медиа"), max_length=255)
    mime_type = models.CharField(_("тип файла"), max_length=255)
    extra_info = models.JSONField(_("дополнительная информация"), default=dict)
    folder = TreeForeignKey(
        Folder, verbose_name=_("папка"), on_delete=models.CASCADE, null=True, blank=True
    )

    copyright = models.ForeignKey(
        Copyright,
        verbose_name=_("права"),
        related_name="files",
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    rights_comment = models.TextField(_("комментарий по правам"), blank=True)

    author = models.ForeignKey(
        Author,
        verbose_name=_("автор"),
        related_name="files",
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    uploaded_by = models.ForeignKey(
        User,
        verbose_name=_("кем загружено"),
        related_name="uploaded_files",
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )

    class Meta:
        verbose_name = _("медиа")
        verbose_name_plural = _("медиа")

    def __str__(self):
        return self.filename


class Photo(MediaFile, TimeStampedModel):
    exif = models.JSONField(_("EXIF"), default=dict)
    location = models.ForeignKey(
        Location,
        verbose_name=_("локация"),
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
    )
    geo_coords = models.CharField(_("гео-координаты"), max_length=255, blank=True)
    photo_date = models.DateTimeField(_("дата съёмки"), null=True, blank=True)
    preview = models.ImageField(
        _("превью"),
        max_length=1024,
        upload_to=get_preview_upload_path,
        blank=True,
        null=True,
    )
    detected = models.BooleanField(_("распознано"), default=False)

    class Meta:
        verbose_name = _("фото")
        verbose_name_plural = _("фото")

    def __str__(self):
        return self.filename


class Video(MediaFile, TimeStampedModel):
    exif = models.JSONField(_("EXIF"), default=dict)
    video_date = models.DateTimeField(_("дата съёмки"), null=True, blank=True)

    class Meta:
        verbose_name = _("видео")
        verbose_name_plural = _("видео")

    def __str__(self):
        return self.filename


MODEL_FOR_MEDIA_TYPE = {
    "image": Photo,
    "video": Video,
}


class UserOnPhoto(TimeStampedModel):
    photo = models.ForeignKey(Photo, verbose_name=_("фото"), on_delete=models.CASCADE)
    login = models.CharField(_("логин"), max_length=255)
    match_score = models.FloatField(_("процент совпадения"), blank=True, null=True)
    staff_image_url = models.CharField(_("фото на стаффе"), max_length=255, blank=True)
    verified = models.BooleanField(_("проверено"), default=False)

    class Meta:
        verbose_name = _("пользователь на фото")
        verbose_name_plural = _("пользователи на фото")
        unique_together = ["photo", "login"]

    def __str__(self):
        return self.login


class PhotoDuplicate(TimeStampedModel):
    original = models.ForeignKey(
        Photo,
        verbose_name=_("оригинал фото"),
        related_name=_("фото"),
        on_delete=models.CASCADE,
    )
    duplicates = models.ManyToManyField(Photo, verbose_name=_("дубликаты"))
    chksum = models.CharField(_("хеш сумма"), max_length=255, blank=True)

    class Meta:
        verbose_name = _("фото дубликат")
        verbose_name_plural = _("фото дубликаты")
