from django.db import transaction
from django.db.models import signals
from django.dispatch import receiver
from filetype import filetype

from photoalbum.utils import get_date_from_str, get_exif, get_sha256

from .models import MediaFile, Photo, Video
from .tasks import get_location, get_user_on_photo


@receiver(signals.post_delete, sender=MediaFile)
@receiver(signals.post_delete, sender=Photo)
@receiver(signals.post_delete, sender=Video)
def file_post_delete_handler(instance: MediaFile, **kwargs):
    instance.file.delete(save=False)


@receiver(signals.post_save, sender=Photo)
def location_post_save_handler(instance: Photo, **kwargs):
    if not instance.location and instance.geo_coords:
        signals.post_save.disconnect(location_post_save_handler, sender=Photo)
        get_location(instance.id)
        signals.post_save.connect(location_post_save_handler, sender=Photo)


@receiver(signals.pre_save, sender=MediaFile)
@receiver(signals.pre_save, sender=Photo)
@receiver(signals.pre_save, sender=Video)
def file_pre_save_handler(instance: MediaFile, **kwargs):
    if instance.resource_id:
        return
    instance.size = instance.file.size
    mime_type = filetype.guess_mime(instance.file)
    instance.mime_type = mime_type
    instance.media_type = mime_type.partition("/")[0]
    instance.chksum = get_sha256(instance.file)
    if type(instance) == Photo:
        exif = get_exif(instance.file)
        if exif:
            instance.exif = exif
            instance.photo_date = get_date_from_str(exif.get("date_time"))
            geo_coords = (
                exif.get("gps_longitude"),
                exif.get("gps_latitude"),
            )
            instance.geo_coords = str(geo_coords) if all(geo_coords) else ""


@receiver(signals.post_save, sender=Photo)
def deface_post_save_handler(instance: Photo, **kwargs):
    if not instance.detected:
        transaction.on_commit(lambda: get_user_on_photo.delay(instance.id))
