import logging
from datetime import timedelta
from functools import partial
from itertools import chain
from multiprocessing import cpu_count
from multiprocessing.pool import ThreadPool
from tempfile import TemporaryFile

import requests
from django.conf import settings
from django.core.files import File
from django.db import transaction
from django.utils import timezone

from photoalbum.contrib.yadisk import yadisk
from photoalbum.files.constants import MB
from photoalbum.files.models import (
    MODEL_FOR_MEDIA_TYPE,
    Folder,
    MediaFile,
    Photo,
    Video,
)

logger = logging.getLogger(__name__)


def upload_file(media_id, preview):
    media_object = MediaFile.objects.filter(pk=media_id).first()
    if not media_object:
        logger.error(f"Media_id {media_id} is incorrect")
        return
    model = MODEL_FOR_MEDIA_TYPE.get(media_object.media_type, MediaFile)
    filter_by = dict(preview="") if preview else dict(file="")
    with transaction.atomic():
        media = (
            model.objects.select_for_update(skip_locked=True)
            .filter(pk=media_id, **filter_by)
            .first()
        )
        if not media:
            logger.error(f"Media_id {media_id} is incorrect")
            return
        chunk_size = 10 ** 6
        download_object = "preview" if preview else "file"
        kwargs = dict(preview_crop=True, preview_size="M")
        logger.info(f"{media.path}\nmedia_id: {media.resource_id}")
        try:
            media_info = yadisk.get_resources(
                media.path, fields=download_object, **kwargs
            )
            stream = yadisk.get_file(url=media_info.get(download_object), stream=True)
        except Exception as exc:
            logger.error(f"ERROR media: {media.pk}:\n{media.path}\n{exc}")
            return

        try:
            with TemporaryFile() as temp_file:
                for chunk in stream.iter_content(chunk_size):
                    temp_file.write(chunk)
                field = media.preview if preview else media.file
                field.save(media.filename, File(temp_file))

        except OSError as exc:
            logger.error(exc)
            return


def download_files(*args, **kwargs):
    logger.info("Started loading files")
    pool = ThreadPool(kwargs.get("threads", cpu_count()))
    preview = kwargs.get("preview")
    media_ids = kwargs.get("media_ids")
    if media_ids:
        with transaction.atomic():
            pool.map(
                partial(upload_file, preview=preview),
                media_ids,
            )
            pool.close()
            pool.join()
        return

    model = Photo if preview else MODEL_FOR_MEDIA_TYPE.get(kwargs.get("media_type"))
    media_id = kwargs.get("pk")
    if media_id:
        upload_file(media_id, preview)
        return

    filter_by = (
        dict(preview="")
        if preview
        else dict(
            file="",
            size__lt=kwargs.get("max_size", 100000) * MB,
            size__gt=kwargs.get("min_size", 0) * MB,
        )
    )
    limit = kwargs.get("limit")
    if kwargs.get("force"):
        filter_by = dict()
    if model:
        media_objects = model.objects.filter(**filter_by).values_list("pk", flat=True)[
            :limit
        ]
    else:
        media_objects = list(
            chain(
                Photo.objects.filter(**filter_by).values_list("pk", flat=True),
                Video.objects.filter(**filter_by).values_list("pk", flat=True),
                MediaFile.objects.exclude(
                    media_type__in=list(MODEL_FOR_MEDIA_TYPE.keys())
                )
                .filter(**filter_by)
                .values_list("pk", flat=True),
            )
        )[:limit]
    pool.map(
        partial(upload_file, preview=preview),
        media_objects,
    )
    pool.close()
    pool.join()
    logger.info("Finished loading files")


def delete_old_media(model, **kwargs):
    logger.info("Started removing deleted files")
    old_media = model.objects.filter(
        modified__lt=timezone.now()
        - timedelta(days=settings.YADISK_MEDIA_EXPIRATION_DAYS),
        **kwargs,
    )
    for media in old_media:
        check_and_delete(media)
    logger.info("Finished removing deleted files")


def check_and_delete(media):
    logger.info(f"{media.path}\nmedia_id: {media.resource_id}")
    if not media.resource_id:
        return

    response = yadisk.get_resources(media.path)
    if not response:
        logger.error(f"File {media.path} does not exist anymore")
        media.delete()
        return


def delete_empty_folders():
    logger.info("Started removing empty folders")
    for folder in Folder.objects.all():
        if not MediaFile.objects.filter(
            folder__in=folder.get_descendants(include_self=True)
        ).exists():
            check_and_delete(folder)
    logger.info("Finished removing empty folders")


def upload_file_or_folder(media, loader):
    response = yadisk.get_resources(media.path)
    if not response:
        logger.error(
            f"{media.path} does not exist anymore\n" f"downloading it again..."
        )
        try:
            loader(media)
            logger.info(f"Finished loading {media.path}")
        except requests.exceptions.HTTPError as exc:
            logger.error(exc)
        except Exception as exc:
            logger.error(exc)
