import hashlib
from datetime import datetime

import PIL.Image
from django.utils import timezone
from PIL.ExifTags import GPSTAGS, TAGS

from photoalbum.files.constants import DATETIME_FORMATS


def get_exif(file):
    # приватный метод, так как публичный не содержит все необходимые поля
    init_exif = PIL.Image.open(file)._getexif()
    if not init_exif:
        return
    exif = dict()
    for k, v in init_exif.items():
        if TAGS.get(k) == "DateTime":
            exif["date_time"] = v
            break

    coords = get_coordinates(init_exif)
    if coords:
        exif["gps_latitude"], exif["gps_longitude"] = coords
    return exif


def get_date_from_str(date_time):
    if not isinstance(date_time, str) or not date_time:
        return date_time
    for format in DATETIME_FORMATS:
        try:
            date_time = timezone.make_aware(
                datetime.strptime(date_time, format),
                timezone.get_default_timezone(),
            )
            break
        except ValueError:
            continue
    return date_time


def get_decimal_from_dms(dms, ref):
    if not dms:
        return
    degrees = dms[0]
    minutes = dms[1] / 60.0
    seconds = dms[2] / 3600.0

    if ref in ["S", "W"]:
        degrees = -degrees
        minutes = -minutes
        seconds = -seconds

    return round(degrees + minutes + seconds, 6)


def get_coordinates(exif):
    geotags = dict()
    for idx, tag in TAGS.items():
        if tag == "GPSInfo":
            if idx not in exif:
                break
            for key, val in GPSTAGS.items():
                if key in exif[idx]:
                    geotags[val] = exif[idx][key]
    lat = get_decimal_from_dms(
        geotags.get("GPSLatitude"), geotags.get("GPSLatitudeRef")
    )
    lon = get_decimal_from_dms(
        geotags.get("GPSLongitude"), geotags.get("GPSLongitudeRef")
    )
    if not (lat and lon):
        return ""
    return lat, lon


def get_sha256(file):
    return hashlib.sha256(file.read()).hexdigest()
