package ru.yandex.qe.dispenser.api.util;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;

@ParametersAreNonnullByDefault
public final class NumberUtils {
    private NumberUtils() {
    }

    public static long requireNotNegative(final long number) {
        if (number < 0) {
            throw new IllegalArgumentException("Number '" + number + "' is negative, but non negative required!");
        }
        return number;
    }

    public static int requireNotNegative(final int number) {
        return Math.toIntExact(requireNotNegative((long) number));
    }

    public static long requirePositive(final long value, final String valueName) {
        if (value <= 0) {
            throw new IllegalArgumentException(valueName + " must be positive");
        }
        return value;
    }

    public static long requirePositive(final long value) {
        return requirePositive(value, "Number " + value);
    }

    public static int requirePositive(final int number) {
        return Math.toIntExact(requirePositive((long) number));
    }

    public static double requirePositive(final double x) {
        if (x <= 0) {
            throw new IllegalArgumentException("Number '" + x + "' must be positive!");
        }
        return x;
    }

    public static double requireInRange(final double x, final double l, final double r) {
        if (x < l || r < x) {
            throw new IllegalArgumentException("Number '" + x + "' must be in range [" + l + ", " + r + "]");
        }
        return x;
    }

    public static long requireInRange(final long value, final long leftBorder, final long rightBorder) {
        return requireInRange(value, leftBorder, rightBorder, "Number " + value);
    }

    public static long requireInRange(final long value, final long leftBorder, final long rightBorder, final String valueName) {
        if (value < leftBorder || rightBorder < value) {
            throw new IllegalArgumentException("'" + valueName + "' must be in range [" + leftBorder + ", " + rightBorder + "]");
        }
        return value;
    }

    public static long multiple(final long a, final long b, final int times) {
        long result = a;
        for (int i = 0; i < times; i++) {
            result *= b;
        }
        for (int i = 0; i > times; i--) {
            result /= b;
        }
        return result;
    }

    public static double multiple(final double a, final double b, final int times) {
        double result = a;
        for (int i = 0; i < times; i++) {
            result *= b;
        }
        for (int i = 0; i > times; i--) {
            result /= b;
        }
        return result;
    }

    public static long multipleInteger(final long value, final long base, final int times) {
        if (times == 0) {
            return value;
        }
        if (times > 0) {
            final BigInteger scale = BigInteger.valueOf(base).pow(times);
            return BigInteger.valueOf(value).multiply(scale).longValue();
        }
        final BigInteger scale = BigInteger.valueOf(base).pow(Math.abs(times));
        return BigInteger.valueOf(value).divide(scale).longValue();
    }

    @NotNull
    public static BigInteger multipleInteger(@NotNull final BigInteger value, final long base, final int times) {
        if (times == 0) {
            return value;
        }
        if (times > 0) {
            final BigInteger scale = BigInteger.valueOf(base).pow(times);
            return value.multiply(scale);
        }
        final BigInteger scale = BigInteger.valueOf(base).pow(Math.abs(times));
        return value.divide(scale);
    }

    @NotNull
    public static BigDecimal multipleDecimal(@NotNull final BigDecimal value, final long base, final int times,
                                             MathContext mathContext) {
        if (times == 0) {
            return value;
        }
        if (times > 0) {
            final BigDecimal scale = BigDecimal.valueOf(base).pow(times);
            return value.multiply(scale, mathContext);
        }
        final BigDecimal scale = BigDecimal.valueOf(base).pow(Math.abs(times));
        return value.divide(scale, mathContext);
    }
}
