package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.util.Optional;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.NumberUtils;

@JsonSerialize(using = DiAmount.Serializer.class)
@JsonDeserialize(using = DiAmount.Deserializer.class)
public class DiAmount {
    private final long value;
    @NotNull
    private final DiUnit unit;

    protected DiAmount(final long value, @NotNull final DiUnit unit) {
        this.value = NumberUtils.requireNotNegative(value);
        this.unit = unit;
    }

    @NotNull
    public static DiAmount anyOf(@NotNull final DiUnit unit) {
        return of(1, unit);
    }

    @NotNull
    public static DiAmount of(final long value, @NotNull final DiUnit unit) {
        return new DiAmount(value, unit);
    }

    public long getValue() {
        return value;
    }

    @NotNull
    public DiUnit getUnit() {
        return unit;
    }

    @NotNull
    public String getAbbreviation() {
        return unit.getAbbreviation();
    }

    @NotNull
    @SuppressWarnings("ClassReferencesSubclass")
    public DiAmount.Humanized humanize() {
        return DiUnit.humanReadable(this).convertToAmount(Humanized.of((double) getValue(), getUnit()));
    }

    @NotNull
    @SuppressWarnings("ClassReferencesSubclass")
    public static DiAmount.Humanized fromHumanizedString(@NotNull final String string) {
        return DiAmount.Humanized.fromString(string);
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof DiAmount) {
            return value == ((DiAmount) o).value && unit == ((DiAmount) o).unit;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return 31 * ((int) (value ^ (value >>> 32))) + unit.hashCode();
    }

    @NotNull
    @Override
    public String toString() {
        return getValue() + " " + getAbbreviation();
    }

    @JsonSerialize(using = Humanized.Serializer.class)
    public static final class Humanized extends DiAmount {
        private static final DecimalFormat STRING_FORMAT = new DecimalFormat("#.##");

        private final double doubleValue;

        private Humanized(final double doubleValue, @NotNull final DiUnit unit) {
            super(Math.round(doubleValue), unit);
            this.doubleValue = doubleValue;
        }

        @NotNull
        static Humanized of(final double doubleValue, @NotNull final DiUnit unit) {
            return new Humanized(doubleValue, unit);
        }

        @NotNull
        static Humanized fromString(@NotNull final String string) {
            final String[] parts = string.split("\\s+");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Invalid string format");
            }
            final double value = Double.valueOf(parts[0]);
            final DiUnit unit = DiUnit.fromAbbreviation(parts[1]);
            return new Humanized(value, unit);
        }

        public double getDoubleValue() {
            return doubleValue;
        }

        @NotNull
        public String getStringValue() {
            return STRING_FORMAT.format(doubleValue);
        }

        @NotNull
        @Override
        public String toString() {
            return getStringValue() + " " + getAbbreviation();
        }

        static final class Serializer extends JsonSerializerBase<Humanized> {
            @Override
            public void serialize(final @NotNull Humanized value,
                                  @NotNull final JsonGenerator jg,
                                  @NotNull final SerializerProvider sp) throws IOException {
                jg.writeStartObject();
                jg.writeNumberField("doubleValue", value.getDoubleValue());
                jg.writeStringField("stringValue", value.getStringValue());
                jg.writeStringField("unit", value.getUnit().name());
                jg.writeStringField("abbreviation", value.getAbbreviation());
                jg.writeEndObject();
            }
        }
    }

    static final class Serializer extends JsonSerializerBase<DiAmount> {
        @Override
        public void serialize(@NotNull final DiAmount amount,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeNumberField("value", amount.getValue());
            jg.writeStringField("unit", amount.getUnit().name());
            jg.writeObjectFieldStart("humanized");
            DiAmount.Humanized humanized = amount.humanize();
            jg.writeNumberField("doubleValue", humanized.getDoubleValue());
            jg.writeStringField("stringValue", humanized.getStringValue());
            jg.writeStringField("unit", humanized.getUnit().name());
            jg.writeStringField("abbreviation", humanized.getAbbreviation());
            jg.writeEndObject();
            jg.writeEndObject();
        }
    }

    public static final class CompactSerializer extends JsonSerializerBase<DiAmount> {
        @Override
        public void serialize(@NotNull final DiAmount amount,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeNumberField("value", amount.getValue());
            jg.writeStringField("unit", amount.getUnit().name());
            jg.writeEndObject();
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiAmount> {
        @NotNull
        @Override
        public DiAmount deserialize(@NotNull final JsonParser jp,
                                    @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            final JsonNode amountValue = json.get("value");
            final DiUnit amountUnit = DiUnit.valueOf(json.get("unit").asText());
            if (amountValue.isIntegralNumber() && amountValue.canConvertToLong()) {
                // It is an actual integer, just check that it is representable as the smallest unit
                final Optional<DiAmount> normalizedValue = amountUnit.normalize(amountValue.longValue());
                if (normalizedValue.isPresent()) {
                    return of(normalizedValue.get().getValue(), normalizedValue.get().getUnit());
                } else {
                    throw new IllegalArgumentException("Unsupported amount value: " + amountValue.asText());
                }
            } else if (amountValue.isIntegralNumber() && amountValue.isBigInteger()) {
                // It is a huge integer, try to find a suitable unit, check that it is representable as the smallest unit
                final BigInteger value = amountValue.bigIntegerValue();
                final Optional<DiAmount> normalizedValue = amountUnit.normalize(value);
                if (normalizedValue.isPresent()) {
                    return of(normalizedValue.get().getValue(), normalizedValue.get().getUnit());
                } else {
                    throw new IllegalArgumentException("Unsupported amount value: " + amountValue.asText());
                }
            } else if (amountValue.isFloatingPointNumber() && (amountValue.isDouble() || amountValue.isFloat())) {
                // It is a floating point number, try to find a suitable unit, check that it is representable as the smallest unit
                final double value = amountValue.doubleValue();
                final Optional<DiAmount> normalizedValue = amountUnit.normalize(value);
                if (normalizedValue.isPresent()) {
                    return of(normalizedValue.get().getValue(), normalizedValue.get().getUnit());
                } else {
                    throw new IllegalArgumentException("Unsupported amount value: " + amountValue.asText());
                }
            } else if (amountValue.isFloatingPointNumber() && amountValue.isBigDecimal()) {
                // It is a decimal number, try to find a suitable unit, check that it is representable as the smallest unit
                final BigDecimal value = amountValue.decimalValue();
                final Optional<DiAmount> normalizedValue = amountUnit.normalize(value);
                if (normalizedValue.isPresent()) {
                    return of(normalizedValue.get().getValue(), normalizedValue.get().getUnit());
                } else {
                    throw new IllegalArgumentException("Unsupported amount value: " + amountValue.asText());
                }
            } else {
                // Not a numeric value, throw exception
                throw new IllegalArgumentException("Unsupported amount value: " + amountValue.asText());
            }
        }

    }
}
