package ru.yandex.qe.dispenser.api.v1;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;

public class DiBotServiceReserve {
    @Nullable
    private final Long id;
    @NotNull
    private final DiAmount amount;
    @NotNull
    private final Long bigOrderId;
    @NotNull
    private final String resourceKey;
    @NotNull
    private final Set<String> segmentKeys;
    @NotNull
    private final String serviceKey;
    @NotNull
    private final Long campaignId;

    @JsonCreator
    public DiBotServiceReserve(@Nullable @JsonProperty("id") final Long id,
                               @NotNull @JsonProperty("amount") final DiAmount amount,
                               @NotNull @JsonProperty("bigOrderId") final Long bigOrderId,
                               @NotNull @JsonProperty("resourceKey") final String resourceKey,
                               @NotNull @JsonProperty("segmentKeys") final Set<String> segmentKeys,
                               @NotNull @JsonProperty("serviceKey") final String serviceKey,
                               @NotNull @JsonProperty("campaignId") final Long campaignId) {
        this.id = id;
        this.amount = amount;
        this.bigOrderId = bigOrderId;
        this.resourceKey = resourceKey;
        this.segmentKeys = segmentKeys;
        this.serviceKey = serviceKey;
        this.campaignId = campaignId;
    }

    public DiBotServiceReserve(@NotNull final Builder builder) {
        this.amount = ValidationUtils.requireNonNull(builder.amount, "Amount is required!");
        this.bigOrderId = ValidationUtils.requireNonNull(builder.bigOrderId, "Big order id is required!");
        this.resourceKey = ValidationUtils.requireNonNull(builder.resourceKey, "Resource key is required!");
        this.segmentKeys = ValidationUtils.requireNonNull(builder.segmentKeys, "Segment keys is required");
        this.serviceKey = ValidationUtils.requireNonNull(builder.serviceKey, "Service key is required!");
        this.campaignId = ValidationUtils.requireNonNull(builder.campaignId, "Campaign id is required!");
        this.id = builder.id;
    }

    @NotNull
    public DiAmount getAmount() {
        return amount;
    }

    @NotNull
    public Long getBigOrderId() {
        return bigOrderId;
    }

    @NotNull
    public String getResourceKey() {
        return resourceKey;
    }

    @Nullable
    public Long getId() {
        return id;
    }

    @NotNull
    public Set<String> getSegmentKeys() {
        return segmentKeys;
    }

    @NotNull
    public String getServiceKey() {
        return serviceKey;
    }

    @NotNull
    public Long getCampaignId() {
        return campaignId;
    }

    public static Builder forResource(@NotNull final String resourceKey) {
        return new Builder(resourceKey);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DiBotServiceReserve reserve = (DiBotServiceReserve) o;
        return Objects.equals(id, reserve.id) &&
                amount.equals(reserve.amount) &&
                bigOrderId.equals(reserve.bigOrderId) &&
                resourceKey.equals(reserve.resourceKey) &&
                segmentKeys.equals(reserve.segmentKeys) &&
                serviceKey.equals(reserve.serviceKey) &&
                campaignId.equals(reserve.campaignId);
    }

    public static final class Builder implements DtoBuilder<DiBotServiceReserve> {
        private Long id;
        private DiAmount amount;
        private Long bigOrderId;
        private String resourceKey;
        private Set<String> segmentKeys;
        private String serviceKey;
        private Long campaignId;

        private Builder(final String resourceKey) {
            this.resourceKey = resourceKey;
        }

        public Builder withAmount(final DiAmount amount) {
            this.amount = amount;
            return this;
        }

        public Builder forService(final String serviceKey) {
            this.serviceKey = serviceKey;
            return this;
        }

        public Builder withId(final Long id) {
            this.id = id;
            return this;
        }

        public Builder withBigOrder(final Long bigOrderId) {
            this.bigOrderId = bigOrderId;
            return this;
        }

        public Builder withSegments(final String ...segmentKeys) {
            this.segmentKeys = new HashSet<>(Arrays.asList(segmentKeys));
            return this;
        }

        public Builder withSegments(final Set<String> segmentKeys) {
            this.segmentKeys = segmentKeys;
            return this;
        }

        public Builder withCampaignId(final Long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        @NotNull
        @Override
        public DiBotServiceReserve build() {
            return new DiBotServiceReserve(this);
        }
    }
}
