package ru.yandex.qe.dispenser.api.v1;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.ValidationUtils;

public class DiCheck {
    @NotNull
    private final String key;
    @NotNull
    private final String projectKey;
    @NotNull
    private final String quotaSpecKey;
    @NotNull
    private final Collection<String> persons;
    @NotNull
    private final DiCheckType type;
    @NotNull
    private final EnumSet<DiCheckNotificationType> notificationTypes;
    @NotNull
    private final DiCheckCompareType compareType;
    @NotNull
    private final DiCheckValue<?> amount;
    @NotNull
    private final Set<String> segments;

    private DiCheck(@NotNull final Builder builder) {
        this.key = ValidationUtils.requireNonNull(builder.key, "Key is required");
        this.projectKey = ValidationUtils.requireNonNull(builder.projectKey, "Project key is required");
        this.quotaSpecKey = ValidationUtils.requireNonNull(builder.quotaSpecKey, "Quota spec key is required");
        this.persons = validatePersons(builder.persons);
        this.type = ValidationUtils.requireNonNull(builder.type, "Type is required");
        this.notificationTypes = ValidationUtils.requireNonNull(builder.notificationTypes, "Notification types are required");
        this.compareType = ValidationUtils.requireNonNull(builder.compareType, "Compare type is required");
        this.amount = ValidationUtils.requireNonNull(builder.amount, "Amount is required");
        this.segments = builder.segments == null ? Collections.emptySet() : builder.segments;
    }

    @NotNull
    private static Collection<String> validatePersons(final Collection<String> persons) {
        ValidationUtils.requireNonNull(persons, "Persons are required");
        if (persons.isEmpty()) {
            throw new IllegalArgumentException("At least one person must be specified");
        }
        return persons;
    }

    @SuppressWarnings("MethodWithTooManyParameters")
    @JsonCreator
    private static DiCheck fromJson(@JsonProperty("key") final String key,
                                    @JsonProperty("projectKey") final String projectKey,
                                    @JsonProperty("quotaSpecKey") final String quotaSpecKey,
                                    @JsonProperty("persons") final Collection<String> persons,
                                    @JsonProperty("type") final DiCheckType type,
                                    @JsonProperty("notificationTypes") final Set<DiCheckNotificationType> notificationTypes,
                                    @JsonProperty("compareType") final DiCheckCompareType compareType,
                                    @JsonProperty("amount") final DiCheckValue<?> amount,
                                    @JsonProperty("segments") final Set<String> segments) {
        return new DiCheck.Builder()
                .setKey(key)
                .setProjectKey(projectKey)
                .setQuotaSpecKey(quotaSpecKey)
                .setPersons(persons)
                .setType(type)
                .setNotificationTypes(EnumSet.copyOf(notificationTypes))
                .setCompareType(compareType)
                .setAmount(amount)
                .setSegments(segments)
                .build();
    }

    @NotNull
    public String getKey() {
        return key;
    }

    @NotNull
    public String getProjectKey() {
        return projectKey;
    }

    @NotNull
    public String getQuotaSpecKey() {
        return quotaSpecKey;
    }

    @NotNull
    public Collection<String> getPersons() {
        return persons;
    }

    @NotNull
    public DiCheckType getType() {
        return type;
    }

    @NotNull
    public EnumSet<DiCheckNotificationType> getNotificationTypes() {
        return notificationTypes;
    }

    @NotNull
    public DiCheckCompareType getCompareType() {
        return compareType;
    }

    @NotNull
    public DiCheckValue<?> getAmount() {
        return amount;
    }

    @NotNull
    public Set<String> getSegments() {
        return segments;
    }

    public static class Builder {
        @Nullable
        private String key;
        @Nullable
        private String projectKey;
        @Nullable
        private String quotaSpecKey;
        @Nullable
        private Collection<String> persons;
        @Nullable
        private DiCheckType type;
        @Nullable
        private EnumSet<DiCheckNotificationType> notificationTypes;
        @Nullable
        private DiCheckCompareType compareType;
        @Nullable
        private DiCheckValue<?> amount;
        @Nullable
        private Set<String> segments;

        @NotNull
        public Builder setProjectKey(@NotNull final String projectKey) {
            this.projectKey = projectKey;
            return this;
        }

        @NotNull
        public Builder setQuotaSpecKey(@NotNull final String quotaSpecKey) {
            this.quotaSpecKey = quotaSpecKey;
            return this;
        }

        @NotNull
        public Builder setPersons(@NotNull final Collection<String> persons) {
            this.persons = persons;
            return this;
        }

        @NotNull
        public Builder setType(@NotNull final DiCheckType type) {
            this.type = type;
            return this;
        }

        @NotNull
        public Builder setNotificationTypes(@NotNull final EnumSet<DiCheckNotificationType> notificationTypes) {
            this.notificationTypes = notificationTypes;
            return this;
        }

        @NotNull
        public Builder setCompareType(@NotNull final DiCheckCompareType compareType) {
            this.compareType = compareType;
            return this;
        }

        @NotNull
        public Builder setAmount(@NotNull final DiCheckValue<?> amount) {
            this.amount = amount;
            return this;
        }

        @NotNull
        public Builder setKey(@NotNull final String key) {
            this.key = key;
            return this;
        }

        @NotNull
        public Builder setSegments(@Nullable final Set<String> segments) {
            this.segments = segments;
            return this;
        }

        @NotNull
        public DiCheck build() {
            return new DiCheck(this);
        }

        @NotNull
        public Body buildBody() {
            return new Body(this);
        }
    }

    public static class Body {
        @NotNull
        private final String quotaSpecKey;
        @NotNull
        private final Collection<String> persons;
        @NotNull
        private final DiCheckType type;
        @NotNull
        private final EnumSet<DiCheckNotificationType> notificationTypes;
        @NotNull
        private final DiCheckCompareType compareType;
        @NotNull
        private final DiCheckValue<?> amount;
        @NotNull
        private final Set<String> segments;

        private Body(@NotNull final Builder builder) {
            this.quotaSpecKey = ValidationUtils.requireNonNull(builder.quotaSpecKey, "Quota spec key is required");
            this.persons = validatePersons(builder.persons);
            this.type = ValidationUtils.requireNonNull(builder.type, "Type is required");
            this.notificationTypes = ValidationUtils.requireNonNull(builder.notificationTypes, "Notification types are required");
            this.compareType = ValidationUtils.requireNonNull(builder.compareType, "Compare type is required");
            this.amount = ValidationUtils.requireNonNull(builder.amount, "Amount is required");
            this.segments = builder.segments == null ? Collections.emptySet() : builder.segments;
        }

        @SuppressWarnings("MethodWithTooManyParameters")
        @JsonCreator
        private static Body fromJson(@JsonProperty("quotaSpecKey") final String quotaSpecKey,
                                     @JsonProperty("persons") final Collection<String> persons,
                                     @JsonProperty("type") final DiCheckType type,
                                     @JsonProperty("notificationTypes") final Set<DiCheckNotificationType> notificationTypes,
                                     @JsonProperty("compareType") final DiCheckCompareType compareType,
                                     @JsonProperty("amount") final DiCheckValue<?> amount,
                                     @JsonProperty("segments") final Set<String> segments) {
            return new DiCheck.Builder()
                    .setQuotaSpecKey(quotaSpecKey)
                    .setPersons(persons)
                    .setType(type)
                    .setNotificationTypes(EnumSet.copyOf(notificationTypes))
                    .setCompareType(compareType)
                    .setAmount(amount)
                    .setSegments(segments)
                    .buildBody();
        }

        @NotNull
        public String getQuotaSpecKey() {
            return quotaSpecKey;
        }

        @NotNull
        public Collection<String> getPersons() {
            return persons;
        }

        @NotNull
        public DiCheckType getType() {
            return type;
        }

        @NotNull
        public EnumSet<DiCheckNotificationType> getNotificationTypes() {
            return notificationTypes;
        }

        @NotNull
        public DiCheckCompareType getCompareType() {
            return compareType;
        }

        @NotNull
        public DiCheckValue<?> getAmount() {
            return amount;
        }

        @NotNull
        public Set<String> getSegments() {
            return segments;
        }
    }
}
