package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.KeyBase;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;

@JsonDeserialize(using = DiEntitySpec.Deserializer.class)
public final class DiEntitySpec extends KeyBase<String> {
    @NotNull
    private final DiService service;
    @NotNull
    private final String description;
    @NotNull
    private final List<String> resourceKeys;

    private DiEntitySpec(@NotNull final Builder builder) {
        super(builder.getKey());
        service = ValidationUtils.requireNonNull(builder.service, "Service is required!");
        description = ValidationUtils.requireNonNull(builder.description, "Description is required!");
        resourceKeys = builder.resourceKeys;
    }

    @NotNull
    public static Builder builder(@NotNull final String key) {
        return new Builder(key);
    }

    @NotNull
    public DiService getService() {
        return service;
    }

    @NotNull
    public String getDescription() {
        return description;
    }

    @NotNull
    public List<String> getResourceKeys() {
        return resourceKeys;
    }

    public static final class Builder extends KeyBase.Builder<String, Builder> implements DtoBuilder<DiEntitySpec> {
        @Nullable
        private DiService service;
        @Nullable
        private String description;
        @NotNull
        private final List<String> resourceKeys = new ArrayList<>();

        private Builder(@NotNull final String key) {
            super(key);
        }

        @NotNull
        public Builder inService(@NotNull final DiService service) {
            this.service = service;
            return this;
        }

        @NotNull
        public Builder withDescription(@NotNull final String description) {
            this.description = description;
            return this;
        }

        @NotNull
        public Builder overResources(@NotNull final String... resourceKeys) {
            return overResources(Arrays.asList(resourceKeys));
        }

        @NotNull
        public Builder overResources(@NotNull final Collection<String> resourceKeys) {
            this.resourceKeys.addAll(resourceKeys);
            return this;
        }

        @NotNull
        @Override
        public DiEntitySpec build() {
            return new DiEntitySpec(this);
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiEntitySpec> {
        @NotNull
        @Override
        public DiEntitySpec deserialize(@NotNull final JsonParser jp, @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            return builder(json.get("key").asText())
                    .inService(SerializationUtils.convertValue(json.get("service"), DiService.class))
                    .withDescription(json.get("description").asText())
                    .overResources(SerializationUtils.convertValue(json.get("resourceKeys"), String[].class))
                    .build();
        }
    }
}
