package ru.yandex.qe.dispenser.api.v1;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Function;

import com.fasterxml.jackson.databind.JsonNode;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.NumberUtils;

public final class DiMetaField<T> {
    @NotNull
    private final String key;
    @NotNull
    private final Type<T> type;

    private DiMetaField(@NotNull final String key, final @NotNull Type<T> type) {
        this.key = key;
        this.type = type;
    }

    @NotNull
    public static <T> DiMetaField<T> of(@NotNull final String key, @NotNull final Type<T> type) {
        return new DiMetaField<>(key, type);
    }

    @NotNull
    public String getKey() {
        return key;
    }

    @NotNull
    public Type<T> getType() {
        return type;
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        return key.equals(((DiMetaField<?>) o).key);
    }

    @Override
    public int hashCode() {
        return key.hashCode();
    }

    public static final class Type<T> {
        private static final Set<Type<?>> REGISTRY = new HashSet<>();

        public static final Type<String> STRING = new Type<>("String", String.class, JsonNode::asText);

        public static final Type<Boolean> BOOLEAN = new Type<>("Boolean", Boolean.class, JsonNode::asBoolean);

        public static final Type<Integer> INTEGER = new Type<>("Integer", Integer.class, JsonNode::asInt);

        public static final Type<Long> LONG = new Type<>("Long", Long.class, JsonNode::asLong);
        public static final Type<Long> POSITIVE_LONG = new Type<>("PositiveLong", Long.class, node -> NumberUtils.requirePositive(node.asLong()));

        public static final Type<Double> DOUBLE = new Type<>("Double", Double.class, JsonNode::asDouble);
        public static final Type<Double> POSITIVE_DOUBLE = new Type<>("PositiveDouble", Double.class, node -> NumberUtils.requirePositive(node.asDouble()));
        public static final Type<Double> ZERO_ONE_DOUBLE = new Type<>("ZeroOneDouble", Double.class, node -> NumberUtils.requireInRange(node.asDouble(), 0, 1));

        @NotNull
        private final String name;
        @NotNull
        private final Class<T> dataClass;
        @NotNull
        private final Function<JsonNode, ? extends T> parser;

        private Type(@NotNull final String name, @NotNull final Class<T> dataClass, @NotNull final Function<JsonNode, ? extends T> parser) {
            this.name = name;
            this.dataClass = dataClass;
            this.parser = parser;
            if (!REGISTRY.add(this)) {
                throw new IllegalStateException("Meta '" + name + "' already exists!");
            }
        }

        @NotNull
        public static Type<?> byName(@NotNull final String name) {
            return REGISTRY.stream()
                    .filter(t -> t.getName().equalsIgnoreCase(name))
                    .findFirst()
                    .orElseThrow(() -> new IllegalArgumentException("Unknown meta type '" + name + "'!"));
        }

        @NotNull
        public String getName() {
            return name;
        }

        @NotNull
        public Class<T> getDataClass() {
            return dataClass;
        }

        @NotNull
        public T parse(@NotNull final JsonNode value) {
            return parser.apply(value);
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            return name.equals(((Type<?>) o).name);
        }

        @Override
        public int hashCode() {
            return name.hashCode();
        }

        @NotNull
        private static String requireValidKey(@NotNull final String key) {
            return key;
        }
    }
}
