package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.TextUtils;

@JsonSerialize(using = DiPersonGroup.Serializer.class)
@JsonDeserialize(using = DiPersonGroup.Deserializer.class)
public final class DiPersonGroup {
    public static final DiPersonGroup EMPTY = builder().build();

    @NotNull
    private final Set<String> persons;
    @NotNull
    private final Map<DiYandexGroupType, Set<String>> groups;

    private DiPersonGroup(@NotNull final Builder builder) {
        this.persons = builder.persons;
        this.groups = builder.groups;
    }

    @NotNull
    public Set<String> getPersons() {
        return Collections.unmodifiableSet(persons);
    }

    @NotNull
    public Set<String> getYandexGroups(@NotNull final DiYandexGroupType type) {
        return Collections.unmodifiableSet(groups.getOrDefault(type, Collections.emptySet()));
    }

    @NotNull
    public Map<DiYandexGroupType, Set<String>> getYandexGroupsByType() {
        return groups;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DiPersonGroup that = (DiPersonGroup) o;
        return Objects.equals(persons, that.persons) &&
                Objects.equals(groups, that.groups);
    }

    @Override
    public int hashCode() {
        return Objects.hash(persons, groups);
    }

    @NotNull
    public static Builder builder() {
        return new Builder();
    }

    @NotNull
    public static final class Builder implements DtoBuilder<DiPersonGroup> {
        @NotNull
        private final Set<String> persons = new HashSet<>();
        @NotNull
        private final Map<DiYandexGroupType, Set<String>> groups = new EnumMap<>(DiYandexGroupType.class);

        @NotNull
        public Builder addPersons(@NotNull final String... logins) {
            return addPersons(Arrays.asList(logins));
        }

        @NotNull
        public Builder addPersons(@NotNull final Collection<String> logins) {
            logins.stream().map(this::requireValidPerson).forEach(persons::add);
            return this;
        }

        @NotNull
        public Builder addYaGroups(final @NotNull DiYandexGroupType type, @NotNull final String... urls) {
            return addYaGroups(type, Arrays.asList(urls));
        }

        @NotNull
        private Builder addYaGroups(final @NotNull DiYandexGroupType type, @NotNull final Collection<String> urls) {
            for (final String url : urls) {
                groups.computeIfAbsent(type, t -> new HashSet<>()).add(requireValidGroup(url));
            }
            return this;
        }

        @NotNull
        @Override
        public DiPersonGroup build() {
            return new DiPersonGroup(this);
        }

        @NotNull
        private String requireValidPerson(@NotNull final String login) {
            if (TextUtils.isBlank(login)) {
                throw new IllegalArgumentException("Login must not be blank!");
            }
            return login;
        }

        @NotNull
        private String requireValidGroup(@NotNull final String url) {
            if (TextUtils.isBlank(url)) {
                throw new IllegalArgumentException("Login must not be blank!");
            }
            return url;
        }
    }

    static final class Serializer extends JsonSerializerBase<DiPersonGroup> {
        @Override
        public void serialize(@NotNull final DiPersonGroup userGroup,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeObjectField("persons", userGroup.getPersons());
            jg.writeObjectFieldStart("yandexGroups");
            for (final DiYandexGroupType type : userGroup.groups.keySet()) {
                jg.writeObjectField(type.getPlural(), userGroup.getYandexGroups(type));
            }
            jg.writeEndObject();
            jg.writeEndObject();
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiPersonGroup> {
        @Override
        public @NotNull DiPersonGroup deserialize(@NotNull final JsonParser jp,
                                                  @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode node = jp.getCodec().readTree(jp);
            final Builder builder = new Builder().addPersons(SerializationUtils.toStrings(node.get("persons")));
            final JsonNode yandexGroups = node.get("yandexGroups");
            if (yandexGroups != null && !yandexGroups.isNull()) {
                yandexGroups.fieldNames().forEachRemaining(plural -> {
                    final DiYandexGroupType type = DiYandexGroupType.fromPlural(plural);
                    if (type != null) {
                        builder.addYaGroups(type, SerializationUtils.toStrings(yandexGroups.get(plural)));
                    }
                });
            }
            return builder.build();
        }
    }
}
