package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;

@JsonDeserialize(using = DiProjectServiceMeta.Deserializer.class)
public final class DiProjectServiceMeta {
    @NotNull
    private final DiProject project;
    @NotNull
    private final DiService service;
    @NotNull
    private final DiMetaValueSet meta;

    private DiProjectServiceMeta(@NotNull final Builder builder) {
        this.project = ValidationUtils.requireNonNull(builder.project, "Project required!");
        this.service = ValidationUtils.requireNonNull(builder.service, "Servuce required!");
        this.meta = builder.meta;
    }

    @NotNull
    public static Builder builder() {
        return new Builder();
    }

    @NotNull
    public DiProject getProject() {
        return project;
    }

    @NotNull
    public DiService getService() {
        return service;
    }

    @NotNull
    public DiMetaValueSet getMeta() {
        return meta;
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DiProjectServiceMeta other = (DiProjectServiceMeta) o;
        return project.equals(other.project) && service.equals(other.service);

    }

    @Override
    public int hashCode() {
        return 31 * project.hashCode() + service.hashCode();
    }

    public static final class Builder implements DtoBuilder<DiProjectServiceMeta> {
        @Nullable
        private DiProject project;
        @Nullable
        private DiService service;
        @NotNull
        private DiMetaValueSet meta = DiMetaValueSet.EMPTY;

        private Builder() {
        }

        @NotNull
        public Builder ofProject(@NotNull final DiProject project) {
            this.project = project;
            return this;
        }

        @NotNull
        public Builder inService(@NotNull final DiService service) {
            this.service = service;
            return this;
        }

        @NotNull
        public Builder withMeta(@NotNull final DiMetaValueSet meta) {
            this.meta = meta;
            return this;
        }

        @NotNull
        @Override
        public DiProjectServiceMeta build() {
            return new DiProjectServiceMeta(this);
        }
    }

    static class Deserializer extends JsonDeserializerBase<DiProjectServiceMeta> {
        @NotNull
        @Override
        public DiProjectServiceMeta deserialize(@NotNull final JsonParser jp, @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            return builder()
                    .ofProject(SerializationUtils.convertValue(json.get("project"), DiProject.class))
                    .inService(SerializationUtils.convertValue(json.get("service"), DiService.class))
                    .withMeta(SerializationUtils.convertValue(json.get("meta"), DiMetaValueSet.class))
                    .build();
        }
    }
}
