package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.TextUtils;
import ru.yandex.qe.dispenser.api.v1.request.DiActionType;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityReference;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityUsage;

@JsonSerialize(using = DiQuotaChangeCause.Serializer.class)
@JsonDeserialize(using = DiQuotaChangeCause.Deserializer.class)
public final class DiQuotaChangeCause {
    @NotNull
    private final Map<String, String> meta;

    private DiQuotaChangeCause(@NotNull final Builder builder) {
        meta = builder.meta;
    }

    @Nullable
    public String get(@NotNull final String key) {
        return meta.get(key);
    }

    @NotNull
    public static Builder builder(@NotNull final DiQuotaChangeCause.Type type) {
        return new Builder(type.name);
    }

    @NotNull
    public static Builder forceAcqireResource() {
        return builder(Type.FORCE_ACQUIRE_RESOURCE);
    }

    @NotNull
    public static Builder releaseResource() {
        return builder(Type.RELEASE_RESOURCE);
    }

    @NotNull
    public static Builder createEntity(@NotNull final DiEntity entity, @NotNull final DiPerformer performer) {
        return builder(Type.CREATE_ENTITY)
                .meta("dimensions", SerializationUtils.writeValueAsString(entity.getDimensions()))
                .entity(entity)
                .performer(performer);
    }

    @NotNull
    public static DiQuotaChangeCause.Builder releaseEntity(@NotNull final DiEntityReference entity) {
        return builder(Type.RELEASE_ENTITY)
                .entity(entity);
    }

    @NotNull
    public static DiQuotaChangeCause.Builder shareEntity(@NotNull final DiEntityUsage entityUsage, @NotNull final DiPerformer performer) {
        final String description = String.format(
                "%s на %d увеличил использование объекта '%s' в квоте проекта %s",
                performer.getLogin(), entityUsage.getUsagesCount(), entityUsage.getEntity().getKey(), performer.getProjectKey()
        );
        return builder(Type.SHARE_ENTITY)
                .entity(entityUsage.getEntity())
                .meta("usages", entityUsage.getUsagesCount() + "")
                .performer(performer);
    }

    @NotNull
    public static DiQuotaChangeCause.Builder releaseEntitySharing(@NotNull final DiEntityUsage entityUsage, @NotNull final DiPerformer performer) {
        final String description = String.format(
                "%s на %d уменьшил использование объекта '%s' в квоте проекта %s",
                performer.getLogin(), entityUsage.getUsagesCount(), entityUsage.getEntity().getKey(), performer.getProjectKey()
        );
        return builder(Type.RELEASE_ENTITY_SHARING)
                .entity(entityUsage.getEntity())
                .meta("usages", -entityUsage.getUsagesCount() + "")
                .performer(performer);
    }

    @NotNull
    public static DiQuotaChangeCause.Builder releaseAllEntitySharings(@NotNull final DiEntityReference entity, @NotNull final DiPerformer performer) {
        return builder(Type.RELEASE_ALL_ENTITY_SHARINGS)
                .entity(entity)
                .performer(performer);
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        return meta.equals(((DiQuotaChangeCause) o).meta);
    }

    @Override
    public int hashCode() {
        return meta.hashCode();
    }

    public static class Type {
        private static final Set<Type> ALL = new HashSet<>();

        public static final Type FORCE_ACQUIRE_RESOURCE = new Type(DiActionType.FORCE_ACQUIRE_RESOURCE);
        public static final Type RELEASE_RESOURCE = new Type(DiActionType.RELEASE_RESOURCE);
        public static final Type CREATE_ENTITY = new Type(DiActionType.CREATE_ENTITY);
        public static final Type RELEASE_ENTITY = new Type(DiActionType.RELEASE_ENTITY);
        public static final Type SHARE_ENTITY = new Type(DiActionType.SHARE_ENTITY);
        public static final Type RELEASE_ENTITY_SHARING = new Type(DiActionType.RELEASE_ENTITY_SHARING);
        public static final Type RELEASE_ALL_ENTITY_SHARINGS = new Type(DiActionType.RELEASE_ALL_ENTITY_SHARINGS);

        @NotNull
        private final String name;

        public Type(@NotNull final String name) {
            TextUtils.requireUpperCase(name, "Type name must be not empty upper case -- " + name);
            this.name = name;
            if (!ALL.add(this)) {
                throw new IllegalStateException("Type must be singleton!");
            }
        }

        public Type(@NotNull final DiActionType actionType) {
            this(actionType.name());
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            return name.equals(((Type) o).name);
        }

        @Override
        public int hashCode() {
            return name.hashCode();
        }
    }

    public static final class Builder {
        private static final String TYPE_KEY = "type";

        @NotNull
        private final Map<String, String> meta = new HashMap<>();

        private Builder(@NotNull final String type) {
            meta(TYPE_KEY, type);
        }

        @NotNull
        public Builder meta(@NotNull final String key, @NotNull final String value) {
            meta.put(key, value);
            return this;
        }

        @NotNull
        public Builder description(@NotNull final String description) {
            return meta("descrption", description);
        }


        @NotNull
        private Builder entity(@NotNull final DiEntityReference entity) {
            return meta("entityKey", entity.getKey()).meta("entitySpec", entity.getSpecificationKey());
        }

        @NotNull
        private Builder performer(@NotNull final DiPerformer performer) {
            return meta("login", performer.getLogin()).meta("project", performer.getProjectKey());
        }

        @NotNull
        public DiQuotaChangeCause build() {
            return new DiQuotaChangeCause(this);
        }
    }

    static final class Serializer extends JsonSerializerBase<DiQuotaChangeCause> {
        @Override
        public void serialize(@NotNull final DiQuotaChangeCause cause,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeObject(cause.meta);
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiQuotaChangeCause> {
        @NotNull
        @Override
        public DiQuotaChangeCause deserialize(@NotNull final JsonParser jp, @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            if (!json.has(Builder.TYPE_KEY)) {
                throw new IllegalStateException("No 'type' field in DiQuotaChangeCause!");
            }
            final DiQuotaChangeCause.Builder builder = new DiQuotaChangeCause.Builder(json.get(Builder.TYPE_KEY).asText());
            json.fieldNames().forEachRemaining(key -> builder.meta(key, json.get(key).asText()));
            return builder.build();
        }
    }
}
