package ru.yandex.qe.dispenser.api.v1;

import java.util.Collections;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.ValidationUtils;

public final class DiQuotaKey {
    @NotNull
    private final String projectKey;
    @NotNull
    private final String serviceKey;
    @NotNull
    private final String resourceKey;
    @NotNull
    private final String quotaSpecKey;
    @NotNull
    private final Set<String> segmentKeys;

    private DiQuotaKey(@NotNull final Builder builder) {
        this.projectKey = ValidationUtils.requireNonNull(builder.projectKey, "Project key is required");
        this.serviceKey = ValidationUtils.requireNonNull(builder.serviceKey, "Service key is required");
        this.resourceKey = ValidationUtils.requireNonNull(builder.resourceKey, "Resource key is required");
        this.quotaSpecKey = ValidationUtils.requireNonNull(builder.quotaSpecKey, "Quota spec key is required");
        this.segmentKeys = ValidationUtils.requireNonNull(builder.segmentKeys, "Segment keys are required");
    }

    @JsonCreator
    public static DiQuotaKey fromJson(final @JsonProperty("serviceKey") String serviceKey,
                                      final @JsonProperty("resourceKey") String resourceKey,
                                      final @JsonProperty("quotaSpecKey") String quotaSpecKey,
                                      final @JsonProperty("projectKey") String projectKey,
                                      final @JsonProperty("segmentKeys") Set<String> segmentKeys) {
        return new Builder()
                .serviceKey(serviceKey)
                .resourceKey(resourceKey)
                .quotaSpecKey(quotaSpecKey)
                .projectKey(projectKey)
                .segmentKeys(segmentKeys)
                .build();
    }

    @NotNull
    public String getProjectKey() {
        return projectKey;
    }

    @NotNull
    public String getServiceKey() {
        return serviceKey;
    }

    @NotNull
    public String getResourceKey() {
        return resourceKey;
    }

    @NotNull
    public String getQuotaSpecKey() {
        return quotaSpecKey;
    }

    @NotNull
    public Set<String> getSegmentKeys() {
        return segmentKeys;
    }

    public static class Builder {
        @Nullable
        private String projectKey;
        @Nullable
        private String serviceKey;
        @Nullable
        private String resourceKey;
        @Nullable
        private String quotaSpecKey;
        @Nullable
        private Set<String> segmentKeys = Collections.emptySet();

        @NotNull
        public Builder projectKey(@NotNull final String projectKey) {
            this.projectKey = projectKey;
            return this;
        }

        @NotNull
        public Builder serviceKey(@NotNull final String serviceKey) {
            this.serviceKey = serviceKey;
            return this;
        }

        @NotNull
        public Builder resourceKey(@NotNull final String resourceKey) {
            this.resourceKey = resourceKey;
            return this;
        }

        @NotNull
        public Builder quotaSpecKey(@NotNull final String quotaSpecKey) {
            this.quotaSpecKey = quotaSpecKey;
            return this;
        }

        @NotNull
        public Builder segmentKeys(@NotNull final Set<String> segmentKeys) {
            this.segmentKeys = segmentKeys;
            return this;
        }

        @NotNull
        public DiQuotaKey build() {
            return new DiQuotaKey(this);
        }
    }
}
