package ru.yandex.qe.dispenser.api.v1;

import java.io.IOException;
import java.util.Objects;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.KeyBase;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;


@JsonDeserialize(using = DiQuotaSpec.Deserializer.class)
public final class DiQuotaSpec extends KeyBase<String> {
    @NotNull
    private final DiResource resource;
    @NotNull
    private final String description;
    @NotNull
    private final DiQuotaType type;

    private DiQuotaSpec(@NotNull final Builder builder) {
        super(builder.getKey());
        resource = ValidationUtils.requireNonNull(builder.resource, "Resource is required");
        description = ValidationUtils.requireNonNull(builder.description, "Description is required");
        type = ValidationUtils.requireNonNull(builder.type, "Type is required");
    }

    public @NotNull DiResource getResource() {
        return resource;
    }

    @NotNull
    public String getDescription() {
        return description;
    }

    public @NotNull DiQuotaType getType() {
        return type;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        final DiQuotaSpec spec = (DiQuotaSpec) o;
        return Objects.equals(getResource(), spec.getResource());
    }

    public static final class Builder extends KeyBase.Builder<String, Builder> {
        private @Nullable DiResource resource;
        @Nullable
        private String description;
        private @Nullable DiQuotaType type;

        public Builder(@NotNull final String key) {
            super(key);
        }

        @NotNull
        public Builder resource(final @NotNull DiResource resource) {
            this.resource = resource;
            return this;
        }

        @NotNull
        public Builder description(@NotNull final String description) {
            this.description = description;
            return this;
        }

        @NotNull
        public Builder type(final @NotNull DiQuotaType type) {
            this.type = type;
            return this;
        }

        public @NotNull DiQuotaSpec build() {
            return new DiQuotaSpec(this);
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiQuotaSpec> {
        @NotNull
        @Override
        public DiQuotaSpec deserialize(@NotNull final JsonParser jp,
                                       @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            return new Builder(json.get("key").asText())
                    .resource(SerializationUtils.convertValue(json.get("resource"), DiResource.class))
                    .description(json.get("description").asText())
                    .type(DiQuotaType.valueOf(json.get("type").asText()))
                    .build();
        }
    }
}
