package ru.yandex.qe.dispenser.api.v1;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.NumberUtils;

/**
 * ATTENTION: changes in this enum can break clients!
 * <p>
 * User: amosov-f
 * Date: 12.05.16
 * Time: 19:45
 */
public enum DiUnit {
    BPS("Bps"), KBPS("kBps"), MBPS("MBps"), GBPS("GBps"), TBPS("TBps"),
    BINARY_BPS("Bps"), KIBPS("KiBps"), MIBPS("MiBps"), GIBPS("GiBps"), TIBPS("TiBps"),

    MPS("Mps"), KMPS("kMps"), MMPS("MMps"), GMPS("GMps"),

    @Deprecated
    CURRENCY("валюты"),

    BYTE("B"), KIBIBYTE("KiB"), MEBIBYTE("MiB"), GIBIBYTE("GiB"), TEBIBYTE("TiB"),

    PERMILLE("‰"), PERCENT("%"), COUNT("units"), KILO("K units"), MEGA("M units"), GIGA("G units"),

    PERMILLE_CORES("‰ cores"), PERCENT_CORES("% cores"), CORES("cores"), KILO_CORES("K cores"), MEGA_CORES("M cores"), GIGA_CORES("G cores"),

    RUBLES("\u20BD"), THOUSAND_RUBLES("K \u20BD"), MILLION_RUBLES("M \u20BD"),

    GIBIBYTE_BASE("GiB"), TEBIBYTE_BASE("TiB"), PEBIBYTE_BASE("PiB"), EXBIBYTE_BASE("EiB");

    @NotNull
    private final String abbreviation;

    DiUnit(@NotNull final String abbreviation) {
        this.abbreviation = abbreviation;
    }

    @NotNull
    public String getAbbreviation() {
        return abbreviation;
    }

    @NotNull
    public static DiUnit fromAbbreviation(@NotNull final String abbreviation) {
        return Arrays.stream(values())
                .filter(unit -> unit.getAbbreviation().equals(abbreviation))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("No unit with specified abbreviation"));
    }

    public void checkConverting(@NotNull final DiUnit sourceUnit) {
        convert(DiAmount.anyOf(sourceUnit));
    }

    public long convert(@NotNull final DiAmount amount) {
        return convert(amount.getValue(), amount.getUnit());
    }

    public boolean isConvertible(@NotNull final DiAmount amount) {
        if (amount.getUnit() == this) {
            return true;
        }
        final Optional<Ensemble> ensemble = findEnsembleOptional(amount.getUnit());
        return ensemble.isPresent();
    }

    public double convert(@NotNull final DiAmount.Humanized amount) {
        return convert(amount.getDoubleValue(), amount.getUnit());
    }

    public double convert(final double sourceValue, @NotNull final DiUnit sourceUnit) {
        return sourceUnit == this ? sourceValue : findEnsemble(sourceUnit).convert(sourceValue, sourceUnit, this);
    }

    public BigDecimal convert(final BigDecimal sourceValue, @NotNull final DiUnit sourceUnit) {
        return sourceUnit == this ? sourceValue : findEnsemble(sourceUnit).convert(sourceValue, sourceUnit, this, MathContext.UNLIMITED);
    }

    public BigDecimal convert(final BigDecimal sourceValue, @NotNull final DiUnit sourceUnit,
                              @NotNull final MathContext mathContext) {
        return sourceUnit == this ? sourceValue : findEnsemble(sourceUnit).convert(sourceValue, sourceUnit, this, mathContext);
    }

    public long convert(final long sourceValue, @NotNull final DiUnit sourceUnit) {
        return sourceUnit == this ? sourceValue : findEnsemble(sourceUnit).convert(sourceValue, sourceUnit, this);
    }

    public long convertInteger(final long sourceValue, @NotNull final DiUnit sourceUnit) {
        return sourceUnit == this ? sourceValue
                : findEnsemble(sourceUnit).convertInteger(sourceValue, sourceUnit, this);
    }

    @NotNull
    public BigInteger convertInteger(final BigInteger sourceValue, @NotNull final DiUnit sourceUnit) {
        return sourceUnit == this ? sourceValue
                : findEnsemble(sourceUnit).convertInteger(sourceValue, sourceUnit, this);
    }

    @NotNull
    public DiAmount convertToAmount(@NotNull final DiAmount amount) {
        return DiAmount.of(convert(amount), this);
    }

    @NotNull
    public DiAmount.Humanized convertToAmount(@NotNull final DiAmount.Humanized amount) {
        return DiAmount.Humanized.of(convert(amount), this);
    }

    public Optional<DiAmount> normalize(final long value) {
        // All units like this
        final Ensemble ensemble = findEnsemble(this);
        // Check that it is exactly representable as the smallest unit
        return Optional.of(DiAmount.of(value, this)).flatMap(a -> filterUnrepresentable(a, ensemble));
    }

    public Optional<DiAmount> normalize(@NotNull final BigInteger value) {
        return normalize(value, true);
    }

    public Optional<DiAmount> normalize(@NotNull final BigInteger value, final boolean checkExactlyRepresentable) {
        // All units like this
        final Ensemble ensemble = findEnsemble(this);
        // My be it is exact long value
        final Optional<Long> exactLong = exactLong(value);
        if (exactLong.isPresent()) {
            if (checkExactlyRepresentable) {
                // Check that it is exactly representable as the smallest unit
                return Optional.of(DiAmount.of(exactLong.get(), this)).flatMap(a -> filterUnrepresentable(a, ensemble));
            } else {
                return Optional.of(DiAmount.of(exactLong.get(), this));
            }
        }
        // Units larger than current
        final List<UnitWithRatio> unitsAfter = ensemble.unitsAfter(this);
        // Try to find next unit such that the value is representable using this unit
        for (final UnitWithRatio unit : unitsAfter) {
            final BigInteger quotient = value.divide(unit.getRatio());
            final Optional<Long> exactLongQuotient = exactLong(quotient);
            if (exactLongQuotient.isPresent()) {
                if (checkExactlyRepresentable) {
                    // Check that it is exactly representable as the smallest unit
                    return Optional.of(DiAmount.of(exactLongQuotient.get(), unit.getUnit())).flatMap(a -> filterUnrepresentable(a, ensemble));
                } else {
                    return Optional.of(DiAmount.of(exactLongQuotient.get(), unit.getUnit()));
                }
            }
        }
        // Value is not representable as long
        return Optional.empty();
    }

    public Optional<DiAmount> normalize(final double value) {
        // Inf and NaN is not representable
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            return Optional.empty();
        }
        final boolean inLongRange = isInLongRange(value);
        final boolean isInteger = value == Math.floor(value);
        // All units like this
        final Ensemble ensemble = findEnsemble(this);
        // Exact integer actually
        if (inLongRange && isInteger) {
            // Check that it is exactly representable as the smallest unit
            return Optional.of(DiAmount.of((long) value, this)).flatMap(a -> filterUnrepresentable(a, ensemble));
        }
        final BigDecimal decimalValue = BigDecimal.valueOf(value);
        return normalizeDecimal(decimalValue, ensemble, inLongRange);
    }

    public Optional<DiAmount> normalize(@NotNull final BigDecimal value) {
        // All units like this
        final Ensemble ensemble = findEnsemble(this);
        // May be it is an exact long value
        final Optional<Long> exactLong = exactLong(value);
        if (exactLong.isPresent()) {
            // Check that it is exactly representable as the smallest unit
            return Optional.of(DiAmount.of(exactLong.get(), this)).flatMap(a -> filterUnrepresentable(a, ensemble));
        }
        final boolean inLongRange = isInLongRange(value);
        return normalizeDecimal(value, ensemble, inLongRange);
    }

    private Optional<DiAmount> normalizeDecimal(@NotNull final BigDecimal value, @NotNull final Ensemble ensemble, final boolean inLongRange) {
        // Value is too large to be represented by long, try larger units
        if (!inLongRange) {
            // Units larger than current
            final List<UnitWithRatio> unitsAfter = ensemble.unitsAfter(this);
            // Try to find next unit such that the value is representable using this unit
            for (final UnitWithRatio unit : unitsAfter) {
                final BigDecimal ratio = new BigDecimal(unit.getRatio());
                final BigDecimal quotient = value.divideToIntegralValue(ratio);
                final Optional<Long> exactLongQuotient = exactLong(quotient);
                if (exactLongQuotient.isPresent()) {
                    // Check that it is exactly representable as the smallest unit
                    return Optional.of(DiAmount.of(exactLongQuotient.get(), unit.getUnit())).flatMap(a -> filterUnrepresentable(a, ensemble));
                }
            }
            return Optional.empty();
        }
        // Value is not exactly representable as long, try smaller units
        final List<UnitWithRatio> unitsBefore = ensemble.unitsBefore(this);
        // Truncate current value
        BigDecimal previousResult = truncate(value);
        DiUnit previousUnit = this;
        // Try to find previous unit such that the value is representable using this unit
        for (final UnitWithRatio unit : unitsBefore) {
            final BigDecimal ratio = new BigDecimal(unit.getRatio());
            final BigDecimal nextUnit = value.multiply(ratio);
            // Value in previous unit is too large to be representable, use truncated previous value
            if (!isInLongRange(nextUnit) || isUnrepresentable(nextUnit, unit.getUnit(), ensemble)) {
                final Optional<Long> exactLongResult = exactLong(previousResult);
                if (exactLongResult.isPresent()) {
                    // Check that it is exactly representable as the smallest unit
                    return Optional.of(DiAmount.of(exactLongResult.get(), previousUnit)).flatMap(a -> filterUnrepresentable(a, ensemble));
                } else {
                    return Optional.empty();
                }
            }
            // Maybe value in previous unit is exactly representable as long
            final Optional<Long> exactLongNextUnit = exactLong(nextUnit);
            if (exactLongNextUnit.isPresent()) {
                // Check that it is exactly representable as the smallest unit
                return Optional.of(DiAmount.of(exactLongNextUnit.get(), unit.getUnit())).flatMap(a -> filterUnrepresentable(a, ensemble));
            }
            // Truncate value in previous unit
            previousResult = truncate(nextUnit);
            previousUnit = unit.getUnit();
        }
        // Maybe value in the smallest unit is exactly representable as long
        final Optional<Long> exactLongResult = exactLong(previousResult);
        if (exactLongResult.isPresent()) {
            // Check that it is exactly representable as the smallest unit
            return Optional.of(DiAmount.of(exactLongResult.get(), previousUnit)).flatMap(a -> filterUnrepresentable(a, ensemble));
        } else {
            return Optional.empty();
        }
    }

    private Optional<Long> exactLong(@NotNull final BigInteger value) {
        try {
            return Optional.of(value.longValueExact());
        } catch (ArithmeticException e) {
            return Optional.empty();
        }
    }

    private Optional<Long> exactLong(@NotNull final BigDecimal value) {
        try {
            return Optional.of(value.longValueExact());
        } catch (ArithmeticException e) {
            return Optional.empty();
        }
    }

    private static final BigDecimal MIN_LONG = BigDecimal.valueOf(Long.MIN_VALUE);
    private static final BigDecimal MAX_LONG = BigDecimal.valueOf(Long.MAX_VALUE);

    private boolean isInLongRange(final double value) {
        return isInLongRange(BigDecimal.valueOf(value));
    }

    private boolean isInLongRange(@NotNull final BigDecimal value) {
        return value.compareTo(MIN_LONG) >= 0 && value.compareTo(MAX_LONG) <= 0;
    }

    private BigDecimal truncate(@NotNull final BigDecimal value) {
        return value.setScale(0, RoundingMode.DOWN);
    }

    private Optional<DiAmount> filterUnrepresentable(@NotNull final DiAmount amount, @NotNull final Ensemble ensemble) {
        final UnitWithRatio smallestUnit = ensemble.smallestUnit(amount.getUnit());
        if (exactLong(BigInteger.valueOf(amount.getValue()).multiply(smallestUnit.getRatio())).isPresent()) {
            return Optional.of(amount);
        }
        return Optional.empty();
    }

    private boolean isUnrepresentable(@NotNull final BigDecimal value, @NotNull final DiUnit unit, @NotNull final Ensemble ensemble) {
        final UnitWithRatio smallestUnit = ensemble.smallestUnit(unit);
        return !exactLong(truncate(value.multiply(new BigDecimal(smallestUnit.getRatio())))).isPresent();
    }

    @NotNull
    private Ensemble findEnsemble(@NotNull final DiUnit sourceUnit) {
        return findEnsembleOptional(sourceUnit)
                .orElseThrow(() -> new IllegalArgumentException("Can't convert '" + sourceUnit + "' to '" + this + "'!"));
    }

    @NotNull
    private Optional<Ensemble> findEnsembleOptional(@NotNull final DiUnit sourceUnit) {
        return Arrays.stream(Ensemble.values())
                .filter(e -> e.canConvert(sourceUnit, this))
                .findFirst();
    }

    @NotNull
    static DiUnit humanReadable(@NotNull final DiAmount amount) {
        final DiUnit humanizedUnit = Arrays.stream(Ensemble.values())
                .map(ensemble -> ensemble.getHumanReadableUnit(amount))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(amount.getUnit());
        Objects.requireNonNull(humanizedUnit, "Stupid IDEA");
        return fixStrangeCases(humanizedUnit);
    }

    public static DiAmount largestIntegerAmount(long value, DiUnit baseUnit) {
        Ensemble ensemble = Ensemble.ensembleByUnit(baseUnit);
        if (value == 0L) {
            DiUnit unit = ensemble.getUnit2deg().entrySet().stream().filter(e -> e.getValue() >= 0)
                    .sorted(Map.Entry.comparingByValue()).limit(1).findFirst().map(Map.Entry::getKey).orElse(baseUnit);
            if (unit.equals(DiUnit.BPS)) {
                return DiAmount.of(0L, DiUnit.MBPS);
            }
            if (unit.equals(DiUnit.BINARY_BPS)) {
                return DiAmount.of(0L, DiUnit.MIBPS);
            }
            if (unit.equals(DiUnit.BYTE)) {
                return DiAmount.of(0L, DiUnit.GIBIBYTE);
            }
            return DiAmount.of(0L, unit);
        }
        List<UnitWithRatio> unitsAfter = ensemble.unitsAfter(baseUnit);
        for (int i = unitsAfter.size() - 1; i >= 0; i--) {
            UnitWithRatio unitWithRatio = unitsAfter.get(i);
            BigInteger ratio = unitWithRatio.getRatio();
            BigDecimal[] divideAndReminder = BigDecimal.valueOf(value).divideAndRemainder(new BigDecimal(ratio));
            if (divideAndReminder[1].compareTo(BigDecimal.ZERO) == 0) {
                DiUnit targetUnit = unitWithRatio.getUnit();
                return DiAmount.of(divideAndReminder[0].longValueExact(), targetUnit);
            }
        }
        return DiAmount.of(value, baseUnit);
    }

    @NotNull
    private static DiUnit fixStrangeCases(@NotNull final DiUnit humanizedUnit) {
        if (humanizedUnit.isPartOfCount()) {
            return COUNT;
        }

        if (humanizedUnit == PERMILLE_CORES || humanizedUnit == PERCENT_CORES) {
            return CORES;
        }

        return humanizedUnit;
    }

    private boolean isPartOfCount() {
        return this == PERMILLE || this == PERCENT;
    }

    boolean canBeUsedForHumanReadable() {
        return fixStrangeCases(this) == this;
    }

    public static final class Ensemble {
        private static final long MAX_HUMAN_READABLE = 1000;

        private static final Ensemble BYTE_ENSEMBLE = new Ensemble(1024, BYTE.name())
                .put(BYTE, 0)
                .put(KIBIBYTE, 1)
                .put(MEBIBYTE, 2)
                .put(GIBIBYTE, 3)
                .put(TEBIBYTE, 4);

        private static final Ensemble COUNT_ENSEMBLE = new Ensemble(10, COUNT.name())
                .put(PERMILLE, -3)
                .put(PERCENT, -2)
                .put(COUNT, 0)
                .put(KILO, 3)
                .put(MEGA, 6)
                .put(GIGA, 9);

        private static final Ensemble BPS_ENSEMBLE = new Ensemble(10, BPS.name())
                .put(BPS, 0)
                .put(KBPS, 3)
                .put(MBPS, 6)
                .put(GBPS, 9)
                .put(TBPS, 12);

        private static final Ensemble BINARY_BPS_ENSEMBLE = new Ensemble(1024, BINARY_BPS.name())
                .put(BINARY_BPS, 0)
                .put(KIBPS, 1)
                .put(MIBPS, 2)
                .put(GIBPS, 3)
                .put(TIBPS, 4);

        private static final Ensemble PROCESSOR_ENSEMBLE = new Ensemble(10, CORES.name())
                .put(PERMILLE_CORES, -3)
                .put(PERCENT_CORES, -2)
                .put(CORES, 0)
                .put(KILO_CORES, 3)
                .put(MEGA_CORES, 6)
                .put(GIGA_CORES, 9);

        private static final Ensemble RUBLES_ENSEMBLE = new Ensemble(10, RUBLES.name())
                .put(RUBLES, 0)
                .put(THOUSAND_RUBLES, 3)
                .put(MILLION_RUBLES, 6);

        private static final Ensemble CURRENCY_ENSEMBLE = new Ensemble(10, CURRENCY.name())
                .put(CURRENCY, 0);

        private static final Ensemble MPS_ENSEMBLE = new Ensemble(10, MPS.name())
                .put(MPS, 0)
                .put(KMPS, 3)
                .put(MMPS, 6)
                .put(GMPS, 9);

        private static final Ensemble GIBIBYTE_BASE_ENSEMBLE = new Ensemble(1024, GIBIBYTE_BASE.name())
                .put(GIBIBYTE_BASE, 3)
                .put(TEBIBYTE_BASE, 4)
                .put(PEBIBYTE_BASE, 5)
                .put(EXBIBYTE_BASE, 6);


        private final long base;
        @NotNull
        private final String key;
        @NotNull
        private final Map<DiUnit, Integer> unit2deg = new EnumMap<>(DiUnit.class);
        @NotNull
        private final NavigableMap<Integer, DiUnit> deg2unit = new TreeMap<>();

        private Ensemble(final long base, final String key) {
            this.base = base;
            this.key = key;
        }

        @NotNull
        private Ensemble put(@NotNull final DiUnit unit, final int deg) {
            unit2deg.put(unit, deg);
            deg2unit.put(deg, unit);
            return this;
        }

        private boolean canConvert(@NotNull final DiUnit from, @NotNull final DiUnit to) {
            return unit2deg.containsKey(from) && unit2deg.containsKey(to);
        }

        private long convert(final long value, @NotNull final DiUnit from, @NotNull final DiUnit to) {
            return NumberUtils.multiple(value, base, unit2deg.get(from) - unit2deg.get(to));
        }

        private BigDecimal convert(final BigDecimal value, @NotNull final DiUnit from, @NotNull final DiUnit to,
                                   final MathContext mathContext) {
            return NumberUtils.multipleDecimal(value, base, unit2deg.get(from) - unit2deg.get(to), mathContext);
        }

        private double convert(final double value, @NotNull final DiUnit from, @NotNull final DiUnit to) {
            return NumberUtils.multiple(value, base, unit2deg.get(from) - unit2deg.get(to));
        }

        private long convertInteger(final long value, @NotNull final DiUnit from, @NotNull final DiUnit to) {
            return NumberUtils.multipleInteger(value, base, unit2deg.get(from) - unit2deg.get(to));
        }

        @NotNull
        private BigInteger convertInteger(@NotNull final BigInteger value, @NotNull final DiUnit from, @NotNull final DiUnit to) {
            return NumberUtils.multipleInteger(value, base, unit2deg.get(from) - unit2deg.get(to));
        }

        private List<UnitWithRatio> unitsBefore(@NotNull final DiUnit unit) {
            final int deg = unit2deg.get(unit);
            return unit2deg.entrySet().stream().sorted(Map.Entry.<DiUnit, Integer>comparingByValue().reversed()).filter(e -> e.getValue() < deg)
                    .map(e -> new UnitWithRatio(e.getKey(), getRatio(deg, e.getValue()))).collect(Collectors.toList());
        }

        private List<UnitWithRatio> unitsAfter(@NotNull final DiUnit unit) {
            final int deg = unit2deg.get(unit);
            return unit2deg.entrySet().stream().sorted(Map.Entry.comparingByValue()).filter(e -> e.getValue() > deg)
                    .map(e -> new UnitWithRatio(e.getKey(), getRatio(e.getValue(), deg))).collect(Collectors.toList());
        }

        private UnitWithRatio smallestUnit(@NotNull final DiUnit unit) {
            final int deg = unit2deg.get(unit);
            final Map.Entry<DiUnit, Integer> smallestUnit = unit2deg.entrySet().stream().min(Map.Entry.comparingByValue()).get();
            return new UnitWithRatio(smallestUnit.getKey(), getRatio(deg, smallestUnit.getValue()));
        }

        private BigInteger getRatio(final int degLeft, final int degRight) {
            if (degLeft >= degRight) {
                return BigInteger.valueOf(base).pow(degLeft - degRight);
            } else {
                return BigInteger.valueOf(base).pow(degRight - degLeft);
            }
        }

        @Nullable
        private DiUnit getHumanReadableUnit(@NotNull final DiAmount amount) {
            Integer deg = unit2deg.get(amount.getUnit());
            if (deg == null) {
                return null;
            }
            for (long value = amount.getValue(); value > Math.max(base, MAX_HUMAN_READABLE); value /= base) {
                deg++;
            }
            return Optional.ofNullable(deg2unit.ceilingEntry(deg)).orElse(deg2unit.lastEntry()).getValue();
        }

        private boolean hasUnit(DiUnit unit) {
            return unit2deg.containsKey(unit);
        }

        private static Ensemble ensembleByUnit(DiUnit unit) {
            return Arrays.stream(values()).filter(e -> e.hasUnit(unit)).findFirst().orElseThrow();
        }

        @NotNull
        public Map<DiUnit, Integer> getUnit2deg() {
            return unit2deg;
        }

        public long getBase() {
            return base;
        }

        @NotNull
        public String getKey() {
            return key;
        }

        @NotNull
        public static Ensemble[] values() {
            return new Ensemble[]{BYTE_ENSEMBLE, COUNT_ENSEMBLE, BPS_ENSEMBLE, BINARY_BPS_ENSEMBLE, PROCESSOR_ENSEMBLE, RUBLES_ENSEMBLE, CURRENCY_ENSEMBLE, MPS_ENSEMBLE, GIBIBYTE_BASE_ENSEMBLE};
        }


    }

    private static class UnitWithRatio {

        private final DiUnit unit;
        private final BigInteger ratio;

        private UnitWithRatio(final DiUnit unit, final BigInteger ratio) {
            this.unit = unit;
            this.ratio = ratio;
        }

        public DiUnit getUnit() {
            return unit;
        }

        public BigInteger getRatio() {
            return ratio;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final UnitWithRatio that = (UnitWithRatio) o;
            return unit == that.unit
                    && Objects.equals(ratio, that.ratio);
        }

        @Override
        public int hashCode() {
            return Objects.hash(unit, ratio);
        }

    }

}
