package ru.yandex.qe.dispenser.api.v1.field;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;

/**
 * Field of a some DTO. Consists of a field key and a deserialization implementation.
 *
 * @param <T> Type of the field value.
 */
public interface DiField<T> {
    DiField<String> KEY = new StringField("key");
    DiField<String> NAME = new StringField("name");
    DiField<String> DESCRIPTION = new StringField("description");

    @NotNull
    String getKey();

    T fromJson(@NotNull final JsonNode jsonNode);

    abstract class FieldImpl<T> implements DiField<T> {
        @NotNull
        private final String key;

        protected FieldImpl(@NotNull final String key) {
            this.key = key;
        }

        @NotNull
        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String toString() {
            return "FieldImpl{" +
                    "key='" + key + '\'' +
                    '}';
        }
    }

    class StringField extends FieldImpl<String> {

        protected StringField(final @NotNull String key) {
            super(key);
        }

        @Override
        public String fromJson(final @NotNull JsonNode jsonNode) {
            return jsonNode.asText();
        }
    }

    class TypedField<T> extends FieldImpl<T> {

        @NotNull
        private final Class<T> typeClass;

        protected TypedField(final @NotNull String key, @NotNull final Class<T> typeClass) {
            super(key);
            this.typeClass = typeClass;
        }

        @Override
        public T fromJson(final @NotNull JsonNode jsonNode) {
            return SerializationUtils.convertValue(jsonNode, typeClass);
        }
    }

    class TypeReferencedField<T> extends FieldImpl<T> {

        @NotNull
        private final TypeReference<T> typeReference;

        protected TypeReferencedField(final @NotNull String key, @NotNull final TypeReference<T> typeReference) {
            super(key);
            this.typeReference = typeReference;
        }

        @Override
        public T fromJson(final @NotNull JsonNode jsonNode) {
            return SerializationUtils.convertValue(jsonNode, typeReference);
        }
    }
}
